/*
 * mrf_utils.cpp
 *
 *  Created on: 22 Nov 2010
 *      Author: torsten
 */

#include "mrf_utils.h"
#include "graph_gen.h"

#include <MRF2.1/mrf.h>
#include <MRF2.1/GCoptimization.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <math.h>

#ifdef DEBUG_MRF_UTILS
extern
void	mrf_utils_log	(char*);
static
char	log_txt	[1024];
#define	LOG_MRF_UTILS(s) sprintf(log_txt,s); \
							mrf_utils_log(log_txt);
#define	LOG_MRF_UTILS_2(s,t) sprintf(log_txt,s,t); \
							mrf_utils_log(log_txt);
#define	LOG_MRF_UTILS_3(s,t,u) sprintf(log_txt,s,t,u); \
							mrf_utils_log(log_txt);
#define	LOG_MRF_UTILS_4(s,t,u,v) sprintf(log_txt,s,t,u,v); \
							mrf_utils_log(log_txt);
#define	LOG_MRF_UTILS_5(s,t,u,v,w) sprintf(log_txt,s,t,u,v,w); \
							mrf_utils_log(log_txt);
#endif // DEBUG_MRF_UTILS

namespace GraphGen	{

namespace MRFscope	{

u_int32_t
Utils::m_No_Nds	=	0,
Utils::m_No_Edges
				=	0,
Utils::m_No_Labels
				=	0;

Utils::MRF_ReadState
Utils::m_State	=	Utils::MRF_Readstate_idle;

Node<Q_DataTerm>*
Utils::m_Nodes	=	NULL;

Edge_Ising*
Utils::m_Edges	=	NULL;

const float
Edge_Ising::ISING_MODEL_CONST
				=	2.3f;

float
Edge_Ising::m_G [4];

bool
Utils::m_Loaded	=	false;

bool
Utils::m_Init_Called
				= 	false;

const float
Utils::COST_SCALAR
				=	100.;

void
Edge_Ising::Apply_IsingModel() {
	m_G[0] = m_G[3] = 0.;
	m_G[1] = m_G[2] = ISING_MODEL_CONST;
}

static
bool readline(FILE* f) {
	char buf[1024];
	memset((void*)buf,0,sizeof(buf));
	char* c(buf);
	strcpy(c,"\033[32m");
	do {
		*c++ = fgetc(f);

	} while(!feof(f) && *(c-1) != '\n');
	strcpy(c-1,"\033[0m\n");
#ifdef DEBUG_MRF_UTILS
	LOG_MRF_UTILS_2("%s",buf);
#endif //DEBUG_MRF_UTILS

	return (!feof(f));
}

void
Utils::Init() {
    srand(time(NULL));
	m_Nodes = NULL;
	m_Edges = NULL;
	m_Init_Called = true;
}

bool
Utils::Read_MRFspec(const char* filename) {

	m_Loaded = false;
	u_int32_t no(0), no_read(0), cu_rec(0), no1(0), no2(0);
	FILE* f(NULL);
	struct stat buf;
	int ret(0);
	float cost(0.);

	assert(m_Init_Called);

	if(stat(filename,&buf) == 0) {
		f = fopen(filename,"r");
		if(f != NULL) {
			char c(0);
			while(!feof(f) && ret != EOF) {
				c = fgetc(f);
				if(c == '\n') {
#ifdef DEBUG_MRF_UTILS
					LOG_MRF_UTILS("\n");
#endif //DEBUG_MRF_UTILS
					continue;
				} else if(c != '#') {
					fseek (f,-1,SEEK_CUR);
					switch(m_State) {

						case MRF_Readstate_idle:
							ret = fscanf(f,"%u %u %u",&m_No_Nds,&m_No_Edges,&m_No_Labels);
#ifdef DEBUG_MRF_UTILS
							LOG_MRF_UTILS_4("%u %u %u",m_No_Nds,m_No_Edges,m_No_Labels);
#endif //DEBUG_MRF_UTILS
							if(ret != EOF) {
								m_State = MRF_Readstate_nodes;
								m_Nodes	= new Node<Q_DataTerm>[m_No_Nds];
								m_Edges = new Edge_Ising[m_No_Edges];
								memset((void*)m_Nodes,0,sizeof(Node<Q_DataTerm>) * m_No_Nds);
								memset((void*)m_Edges,0,sizeof(Edge_Ising) * m_No_Edges);
								no_read = 0;
							}
							break;

						case MRF_Readstate_nodes:
							ret = fscanf(f,"%u",&cu_rec);
#ifdef DEBUG_MRF_UTILS
							LOG_MRF_UTILS_2("%u ",cu_rec);
#endif // DEBUG_MRF_UTILS

							assert(cu_rec > 0);
							cu_rec--;
							m_Nodes[cu_rec].m_Id = cu_rec;
							no = 0;
							while(!feof(f) && ret != EOF && no < m_No_Labels) {
								ret = fscanf(f,"%f",&m_Nodes[cu_rec].m_Nd.m_Q[no++]);
#ifdef DEBUG_MRF_UTILS
								LOG_MRF_UTILS_2("%.3f ",m_Nodes[cu_rec].m_Nd.m_Q[no-1]);
#endif // DEBUG_MRF_UTILS
							}
							if(++no_read == m_No_Nds) {
								m_State = MRF_Readstate_edges;
								no_read = 0;
								m_Edges[no_read].Apply_IsingModel();
							}
							break;

						case MRF_Readstate_edges:
							ret = fscanf(f,"%u;%u",&no1,&no2);
#ifdef DEBUG_MRF_UTILS
							LOG_MRF_UTILS_3("%u;%u ",no1,no2);
#endif //DEBUG_MRF_UTILS
							assert(no1 > 0 && no2 > 0 && ret == 2);
							no = 0;
							while(!feof(f) && ret != EOF && no < m_No_Labels * m_No_Labels) {
								fscanf(f,"%f",&cost);
								m_Edges[no_read].m_G[no++] = cost;
#ifdef DEBUG_MRF_UTILS
								//LOG_MRF_UTILS_2("%f",cost);
#endif //DEBUG_MRF_UTILS
							}
							{
								m_Edges[no_read].m_Node_1 = &m_Nodes[no1-1];
								m_Edges[no_read].m_Node_2 = &m_Nodes[no2-1];
#ifdef DEBUG_MRF_UTILS
								LOG_MRF_UTILS_5("%.3f %.3f %.3f %.3f",m_Edges[no_read].m_G[0],
																	m_Edges[no_read].m_G[1],
																	m_Edges[no_read].m_G[2],
																	m_Edges[no_read].m_G[3]);
#endif //DEBUG_MRF_UTILS
							}
							if(++no_read == m_No_Edges) {
								m_State = MRF_Readstate_idle;
								m_Loaded = true;
							}
							break;
					}
				} else
					/* Kommentarzeile ueberlesen */
					readline(f);
			}
		}
		fclose(f);
	}

	return m_Loaded;
}

void
Utils::Clean_MRFspec() {
	if(m_Loaded) {
		delete [] m_Nodes;
		delete [] m_Edges;
		m_Nodes = NULL;
		m_Edges = NULL;
		m_Loaded = false;
	}
}

MRF::CostVal
Utils::DateCost_MRFspec(int node, MRF::Label label) {
	MRF::CostVal ret(0);
	if((u_int32_t)node < m_No_Nds && (u_int32_t)label < m_No_Labels)
		ret = (MRF::CostVal)ceil(m_Nodes[node].m_Nd.m_Q[label] * Utils::COST_SCALAR);
	else {
#ifdef DEBUG_MRF_UTILS
		LOG_MRF_UTILS_3("\033[31m node/label %d %d not found.\033[0m\n",node,label);
#endif //DEBUG_MRF_UTILS
	}

	return ret;
}

MRF::CostVal
Utils::SmoothCostGeneralFn(int n1, int n2, MRF::Label l1, MRF::Label l2) {
	MRF::CostVal ret(0);
	if((u_int32_t)n1 < m_No_Nds && (u_int32_t)n2 < m_No_Nds && (u_int32_t)l1 < m_No_Labels && (u_int32_t)l2 < m_No_Labels) {
		ret = (MRF::CostVal)ceil(Edge_Ising::m_G[l1*m_No_Labels+l2] * Utils::COST_SCALAR);
	} else {
#ifdef DEBUG_MRF_UTILS
		LOG_MRF_UTILS_5("\033[31m edge/labelpair %d;%d %d,%d not found.\033[0m\n",n1,n2,l1,l2);
#endif //DEBUG_MRF_UTILS
	}

	return ret;
}

void
Utils::Compute_MAP() {
	try	{

		float t;
		EnergyFunction *energy(NULL);
		DataCost *data         = new DataCost(DateCost_MRFspec);
		SmoothnessCost *smooth = new SmoothnessCost(SmoothCostGeneralFn);
		energy    = new EnergyFunction(data,smooth);

		MRF* mrf = new Swap(m_No_Nds,m_No_Labels,energy);

		for (u_int32_t i(0); i <m_No_Edges; i++)
			mrf->setNeighbors(m_Edges[i].m_Node_1->m_Id,m_Edges[i].m_Node_2->m_Id, 1);

		mrf->initialize();
		mrf->clearAnswer();
		mrf->optimize(5,t);  // run for 5 iterations, store time t it took

		MRF::EnergyVal E_smooth = mrf->smoothnessEnergy();
		MRF::EnergyVal E_data   = mrf->dataEnergy();
		printf("Total Energy = %d (Smoothness energy %d, Data Energy %d)\n", E_smooth+E_data,E_smooth,E_data);
		for (u_int32_t n(0); n < m_No_Nds; n++ )
			printf("Label of node %d is %d\n",n, mrf->getLabel(n));
		delete mrf;

	}
		catch (std::bad_alloc) {
		fprintf(stderr, "*** Error: not enough memory\n");
		exit(1);
	}
}

} } // GraphGen::MRFscope
