/*****************************************************************************
 ** Dateiname:    DBAccess.cc
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Bietet den Zugriff auf Sequenzdateien (Implementation)
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 7.10.2002  | Oliver Schoeffmann      | Erzeugung der Datei
 ** 9.11.2002  | Oliver Schoeffmann      | Erweiterung der moeglichen Formate
 **                                      | auf "plain", FASTA und GenBank
 *****************************************************************************/

#include <strstream>
#include <cstdlib>

#include <DBAccess.hh>
using namespace std;

///////////////////////////////////////////////////////////////////////////////
// Public - Methoden
///////////////////////////////////////////////////////////////////////////////

// Konstruktor
//
// Beschreibung: Versucht die uebergebene Datei zu oeffnen. Es wird eine
//               Exception geworfen, wenn das fehlschlaegt.

DBAccess::DBAccess(const char *filename) throw (DBAccessError) {
    ifstrm.open(filename);
    if (ifstrm.fail())
	throw DBAccessError((string)"DBAccess::DBAccess: " + filename +
			    " nicht lesbar!");
} // DBAccess

//---------------------------------------------------------------------------//

// Destruktor
//
// Beschreibung: Die evtl. noch offene Datei wird geschlossen

DBAccess::~DBAccess() {
    ifstrm.close();
}

//---------------------------------------------------------------------------//

// Funktion: nextItem
//
// Beschreibung: s. Headerdatei

DBGen_t *DBAccess::nextItem() throw (DBAccessError) {
    
    char firstline[256];

    // Festellen der Art der Sequenzdatei und Verteilung der Verarbeitung.

    ifstrm.getline(firstline, 255);
    if (ifstrm.eof())
	return 0;
    string word(firstline);
    if (word[0] == '>')
	return fasta(word);
    if (word == "LOCUS")
	return genbank();

    // "plain" wird hier ausgelesen

    DBGen_t *ret = new DBGen_t;
    if (!ret) {
	throw DBAccessError("DBAccess::nextItem: Speicherfehler");
    }
    int c;
    for (unsigned i = 0; i < word.length(); i++)
	if (isalpha(word[i]))
	    ret->sequence += tolower(word[i]);
    while ((c = ifstrm.get()) != EOF)
	if (isalpha(c))
	    ret->sequence += tolower(c);
    return ret;

} // nextItem


///////////////////////////////////////////////////////////////////////////////
// Private - Methoden
///////////////////////////////////////////////////////////////////////////////

// Private - Methode: fasta
//
// Beschreibung: liest die Sequenz aus einer Datei im FASTA-Format aus
//               Bei FASTA ist in der ersten Zeile der Bezeichner und danach 
//               die reine Sequenz. Der Bezeichner beginnt mit '>'.
//
//               >ABC123
//               actgactg.....
//
// Rueckgabewrt: Zeiger auf eine DBGen_t Struktur mit Sequenz und Bezeichner
//
// Exception: Speicherfehler

DBGen_t *DBAccess::fasta(string &name) throw (DBAccessError) {
    DBGen_t *ret = new DBGen_t;
    if (!ret) {
	throw DBAccessError("DBAccess::nextItem: Speicherfehler");
    }
    int c;
    ret->name = name;
    ret->name.erase(0,1);
    while ((c = ifstrm.get()) != EOF && c != '>')
	if (isalpha(c))
	    ret->sequence += tolower(c);
    if (c == '>')
	ifstrm.putback(c);
    return ret;
} // fasta

//---------------------------------------------------------------------------//

// Private - Methode: genbank
//
// Beschreibung: liest die Sequenz aus einer Datei im GenBank-Format aus
//               Dabei sind einige Ankerpunkte wichtig:
//               LOCUS "Bezeichner" "Laenge der Sequenz"
//               CDS Angabe der Exons
//               ORIGIN
//                  "Sequenz"
//               //
//
// Rueckgabewert: Zeiger auf eine DBGen_t Struktur mit Sequenz, Bezeichner,
//                Laenge der Sequenz und annotierter Genstruktur.
//
// Exception: Speicherfehler

DBGen_t *DBAccess::genbank() throw (DBAccessError) {
    string word, line;
    int c;
    DBGenParts_t *akt = 0;
    DBGen_t *ret = new DBGen_t;
    DBGenCDS_t *retp = 0;
    if (!ret) {
	throw DBAccessError("DBAccess::nextItem: Speicherfehler");
    }
    
    // Der Bezeichner und die Laenge werden gelesen
    // "LOCUS" ist bereits entfernt
    ifstrm >> ret->name;
    ifstrm >> ret->length;
    getline(ifstrm, line);
    do {
	ifstrm >> word;
	if (word == "CDS") {
	    // Annotation aus der Datei wird ausgelesen.
	    if (retp) {
		retp->next = new DBGenCDS_t;
		retp = retp->next;
	    } else {
		ret->cds = new DBGenCDS_t;
		retp = ret->cds;
	    }
	    if (!retp)
		throw DBAccessError("DBAccess::nextItem: Speicherfehler");
	    line.erase();
	    ifstrm >> line;
	    if (line.find("complement") == 0) {
		line.erase(0,11);
		retp->trend = -1;
	    } else {
		retp->trend = 1;
	    }
	    if (!isdigit(line[0]) && line.find("join") == string::npos) {
		continue;
	    }
	    if (line.find("join") != string::npos) {
		while (line.find(")") == string::npos) {
		    string dazu;
		    ifstrm >> dazu;
		    line += dazu;
		}
		line.erase(0,5);
		line.erase(line.length() - 1, 1);
	    }
	    istrstream strstrm(line.c_str());

	    // die einzelnen Exons der Annotation werden ausgelesen
	    do {
		if (!retp->parts) {
		    retp->parts = new DBGenParts_t;
		    akt = retp->parts;
		} else {
		    akt->next = new DBGenParts_t;
		    akt = akt->next;
		}
		if (!akt) {
		    delete ret;
		    throw DBAccessError("DBAccessError::nextItem: "
					"Speicherfehler");
		}
		strstrm >> akt->from;
		strstrm.ignore(2);
		strstrm >> akt->to;
	    } while (strstrm.get() == ',');
	}
	getline(ifstrm, line);
    } while (word != "ORIGIN");

    // die eigentliche Sequenz wird ausgelesen
    while ((c = ifstrm.get()) != EOF && c != '/')
	if (isalpha(c))
	    ret->sequence += tolower(c);
    getline(ifstrm, line); // Zeile leeren
    return ret;
} // genbank

