/*****************************************************************************
 ** Dateiname:    DBCombined.cc
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Kombinierte EST- und Proteinverarbeitung
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 2.4.2003   | Oliver Schoeffmann      | Erzeugung der Datei
 *****************************************************************************/

#include <iostream>
#include <cstring>
#include <cstdio>

#include <DBBlast.hh>
#include <DBCombined.hh>
#include <DBStore.hh>

using namespace std;

///////////////////////////////////////////////////////////////////////////////
// Public - Methoden
///////////////////////////////////////////////////////////////////////////////

// Konstruktor
// 
// Beschreibung: s. Headerdatei

DBCombined::DBCombined(DBProperties properties) {
    setProperties(properties);
}

//---------------------------------------------------------------------------//

// Konstruktor
//
// Beschreibung: Einige Variabeln muessen initialisiert werden. Sonst 
// uebernimmt diese Aufgabe 'setProperties'

DBCombined::DBCombined() {
    s_ignore = 0;
    s_has = 0;
    threshold = 0;
    ratio = 0.0;
}

//---------------------------------------------------------------------------//

// Destruktor
//
// Beschreibung: Gibt den Speicher von zwei Variablen wieder frei

DBCombined::~DBCombined() {
    delete s_ignore;
    delete s_has;
}

//---------------------------------------------------------------------------//

// Methode: setProperties
//
// Beschreibung: s.Headerdatei. Initialisiert interne Variablen.

void DBCombined::setProperties(DBProperties properties) {
    props = properties;
    readProps();
}

//---------------------------------------------------------------------------//

// Methode: combinedProcess
//
// Beschreibung: s. Headerdatei

DBMatch_t *DBCombined::combinedProcess(const char *sequence,
				       DBMatch_t *est_matches) 
  throw (DBCombinedError)
{
    if (!sequence)
        throw DBCombinedError("DBCombined::proceed: keine Sequenz angegeben");

    // keine EST-Treffer => keine kombinierten Ergebnisse
    if (!est_matches)
        return 0;

    // 'seq' bietet den Zugriff auf die Sequenz fuer interne Methoden
    seq = sequence;
    DBStore store;
    DBBlast blast(props);
    
    // Einzelne Verarbeitung der EST-Treffer
    for (DBMatch_t *m = est_matches; m; m = m->next) {
        DBMatch_t *estm = 0, *estbm = 0, *blastmatch = 0;
        char *protseq = 0;

	// Markiere die Startpositionen der einzelnen Abschnitte innerhalb
	// der "reinen" EST-Sequenz (s. Beschreibung von 'setOffset') 
        estm = setOffset(m);

	// Erzeugen der "reinen" EST-Sequenz
        protseq = getMatchSequence(estm);
	
	// Durchfuehrung der Datenbankabfrage
        try {
            blastmatch = blast.process(protseq, PROT);
        } catch (DBBlastError &err) {
            throw DBCombinedError((string)"DBCombined::proceedCombined:blast.process(protseq, PROT) "
				  "Fehler produziert durch:\n" 
				  + err.getMessage());
        }
	
	// Bearbeiten der Datenbanktreffer
        for (DBMatch_t *bm = blastmatch; bm; bm = bm->next) {
            if (!namecheck(bm->name))
                continue;

	    // Rueckprojektion auf den EST-Treffer
            estbm = transform(bm, estm);

	    // nur Treffer nehmen, die keinen Schnitt mit bisherigen Treffern
	    // aufweisen
	    if (estbm && !store.cutWith(estbm, false)) {
		try {
		    store.add(estbm, false, true);
		} catch (DBStoreError &err) {
		    throw DBCombinedError((string)
					  "DBCombined::proceedCombined:store.add "
					  "Fehler produziert durch:\n" 
					  + err.getMessage());
		}
	    }
            delete estbm;
        }

	// aufruemen
        delete estm;
        delete blastmatch;
        delete protseq;
    } // for (DBMatch *m ...

    // Rueckgabe der kombinierten Treffer
    DBMatch_t *ret = 0;
    try {
        ret = store.getMatches();
    } catch (DBMatchError &err) {
        throw DBCombinedError((string)"DBCombined::proceedCombined:store.getMatches() "
			      "Fehler produziert durch:\n" 
			      + err.getMessage());
	
    }
    return ret;
} // combinedProcess


///////////////////////////////////////////////////////////////////////////////
// Private - Methoden
///////////////////////////////////////////////////////////////////////////////

// Private - Methode: setOffset
//
// Beschreibung: Diese Methode erzeugt eine Kopie eines Treffers und nutzt
//               die 'sbj_...' Elemete der Teiltreffer-Strukturen um die 
//               Position eines Teiltreffers innerhalb der "reinen" 
//               EST-Sequenz zu markieren.
//               Der erste Abschnitt beginnt dementsprechen mit 1 und endet
//               mit der Laenge des ersten Abschnittes. Der zweite Abschnitt
//               beginnt mit dem letzten Wert des vorherigen Abschnittes + 1, 
//               und endet in der Summe der bisherigen Einzellaengen, usw.
//
// Parameter: 'm' - der zu bearbeitende Treffer
//
// Rueckgabewert: Eine bearbeitete Kopie des Parameters
//
// Exceptions: Fehler beim Kopieren, falscher Aufbau des Parameters

DBMatch_t *DBCombined::setOffset(DBMatch_t *m) throw (DBCombinedError) {
    if (!m || !m->parts)
        throw DBCombinedError("DBCombined::setOffset: "
			      "Fehlerhafter EST-Treffer in der Verarbeitung");
    DBMatch_t *match;
    try {
        match = m->copy();
    } catch (DBMatchError &err) {
        throw DBCombinedError((string)"DBCombined::setOffset: "
			      "Fehler produziert durch:\n" 
			      + err.getMessage());
    }

    // Bestimmung der Positionen
    long offset = 0;
    for (DBParts_t *mp = match->parts; mp; mp = mp->next) {
        mp->sbj_from = offset + 1;
        offset += mp->qry_to - mp->qry_from + 1;
        mp->sbj_to = offset;
    }
    return match;
} //set Offset

//---------------------------------------------------------------------------//

// Private - Methode: getMatchSequence
//
// Beschreibung: Ermittelt zu einem EST-Treffer die "reine" Sequenz.
//               Das ist die Konkatenation der Abschnitte in der Eingabesequenz
//               der einzelnen Teiltreffer.
//
// Parameter: 'm' - EST-Treffer
//
// Rueckgabewert: Zeichenfolge der "reinen" EST-Sequenz
//
// Exception: Falscher Aufbau des Parameters, Speicherfehler

char *DBCombined::getMatchSequence(DBMatch_t *m) throw (DBCombinedError) {
    if (!m || !m->parts)
        throw DBCombinedError("DBCombined::getMatchSequence: "
			      "Fehlerhafter EST-Treffer in der Verarbeitung");
    string ret("");
    for (DBParts_t *p = m->parts; p; p = p->next)
        for (long i = p->qry_from; i <= p->qry_to; i++)
            ret += seq[i - 1];
    char *re = new char[ret.length() + 1];
    if (!re)
        throw DBCombinedError("DBCombined::getMatchSequence: "
			      "Speicherfehler");
    strcpy (re, ret.c_str());
    return re;
} // getMatchSequence

//---------------------------------------------------------------------------//

// Private - Methode: transform
//
// Beschreibung: Projeziert einen Proteintreffer auf den urspruenglichen 
//               EST-Treffer zurueck.
// Bemerkung: Nur der groesste Proteinteiltreffer wird fuer die Bearbeitung 
//            herangezogen.
//
// Parameter: 'bm' - Treffer der Protein-Datenbanksuche
//            'est_m' - Urspruenglicher EST-Treffer.
//
// Rueckgabewert: Kombinierter Treffer
//
// Vorbedingung: In den "sbj_..." Elementen der Teiltreffer von 'est_m' ist das
//               Offset der Teiltreffer durch 'setOffset' gesetzt worden.
//
// Exception: Fehler beim Kopieren, Speicherfehler

DBMatch_t *DBCombined::transform(DBMatch_t *bm, 
				 DBMatch_t *est_m) throw (DBCombinedError) {
    if (!bm || !est_m || !bm->parts || !est_m->parts)
        return 0;
    long score = 0;
    DBParts_t *bigbp = 0;

    // Nur der "beste" Proteinteiltreffer wird fuer die Bearbeitung genommen.
    // Dabei muss das Verh"altnis der Uebereinstimmungen (innerhalb des 
    // Teiltreffers) zur L"ange des Teiltreffers mindestens 'ratio' erreichen.
    // (interne Variable, 0 <= ratio <= 1).
    for (DBParts_t *bp = bm->parts; bp; bp = bp->next) {
        if (((double)bp->akt_id / bp->length) >= ratio && bp->score > score) {
            score = bp->score;
            bigbp = bp;
        }
    }

    // Es wurde kein Teiltreffer gefunden, der die Bedingungen erfuellt, oder
    // der Score dieses Teiltreffers unterschreitet die vorgegebene Grenze
    // (Interne Variable 'threshold')
    if (!bigbp || bigbp->score < threshold)
        return 0;

    // Die Orientierung des Teiltreffers (zur Eingabesequenz) muss mit der 
    // Orientierung (falls vorhanden - != 0) uebereinstimmen
    int trend = est_m->trend == 0 ? bm->trend : est_m->trend;
    if (trend != bm->trend)
	return 0;


    // Der Proteinteiltreffer wird auf den urspruenglichen EST-Treffer
    // zurueckprojeziert (sie werden "kombiniert")
    // Wichtige Daten werden uebernommen
    DBMatch_t *ret = new DBMatch_t;
    if (!ret)
        throw DBCombinedError("DBCombined::transform: Speicherfehler");
    ret->cluster = bm->cluster;
    ret->trend = trend;
    ret->name = new char[strlen(bm->name) + 1];
    if (!ret->name)
        throw DBCombinedError("DBCombined::transform: Speicherfehler");
    strcpy(ret->name, bm->name);

    // Die eigendliche Projektion
    DBParts_t *dbp = est_m->parts, *akt = 0;

    // Suchen der Anfangsposition innerhalb des EST-Treffers
    while (dbp->sbj_to < bigbp->qry_from)
        dbp = dbp->next;
    try {
	ret->parts = dbp->copy();
    } catch (DBMatchError &err) {
	throw DBCombinedError((string)"DBCombined::transform:"
			      + err.getMessage());
    }
    akt = ret->parts;
    akt->qry_from += (bigbp->qry_from - dbp->sbj_from);

    // Setzen der Markierungen start und stopp "vorne"
    // (ist der Trend negativ, so beginnt die Sequenz mit stopp).
    if (trend == 1) {
	if (bigbp->sbj_from == 1 && isStart(akt->qry_from, false))
	    akt->start = true;
    } else {
	if (bigbp->sbj_from == bm->wholeLength && isEnd(akt->qry_from, true)) {
	    akt->end = true;
	    akt->qry_from -= 3;
	}
    }

    // Deht sich der Proteinteiltreffer ueber mehrere EST-Teiltreffer aus, 
    // so werden diese Teile einzeln uebernommen
    while (dbp->next && dbp->next->sbj_from < bigbp->qry_to) {
	dbp = dbp->next;
	try {
	    akt->next = dbp->copy();
	} catch (DBMatchError &err) {
	    throw DBCombinedError((string)"DBCombined::transform:"
				  + err.getMessage());
	}
	akt = akt->next;
    }
    akt->qry_to -= (dbp->sbj_to - bigbp->qry_to);

    // Setzen der Markierungen start und stopp "hinten"
    if (trend == -1) {
	if (bigbp->sbj_to == 1 && isStart(akt->qry_to, true))
	    akt->start = true;
    } else {
	if (bigbp->sbj_to == bm->wholeLength && isEnd(akt->qry_to, false)) {
	    akt->end = true;
	    akt->qry_to += 3;
	}
    }
    
    // setzen des Leserahmens der einzelnen Teiltreffer (kombiniert)
    setFrames(ret);

    return ret;

} // transform

//---------------------------------------------------------------------------//

// Private - Methode: isStart
//
// Beschreibung: Ueberprueft die angegebene Position auf ein Startcodon
//               Die angegebene Position ist dabei das erste Nukleotid 
//               des zu untersuchenden Codons (in Leserichtung des Codons).
//
// Parameter: 'start'   - Startposition des zu untersuchenden Codons
//            'reverse' - Angabe der Leserichtung: 
//                        true  = Vorw"artsstrang,
//                        false = Gegenstrang
//
// Rueckgabewert: true, wenn ein Startcodon in der angegebenen Lererichtung
//                gefunden wurde sonst false.
//
// Bemerkung: Positionsangabe ist NICHT C-like. Daher ist stets eine Position 
//            abzuziehen

bool DBCombined::isStart(long start,
			 bool reverse) {
    if (reverse) {
	if (start > 2)
	    return (seq[start - 1] == 't' && seq[start - 2] == 'a'
		    && seq[start - 3] == 'c');
	else
	    return false;
    }
    if (start < (long)strlen(seq) - 1)
	return (seq[start - 1] == 'a' && seq[start] == 't'
		&& seq[start + 1] == 'g');
    else
	return false;
} // isStart

//---------------------------------------------------------------------------//

// Private - Methode: isEnd
//
// Beschreibung: Ueberprueft die angegebene Position auf ein Stoppcodon
//               Die angegebene Position ist dabei vor dem ersten Nukleotid 
//               des zu untersuchenden Codons (in Leserichtung des Codons).
//
// Parameter: 'end'     - Position vor dem zu untersuchenden Codon
//            'reverse' - Angabe der Leserichtung: 
//                        true  = Vorw"artsstrang,
//                        false = Gegenstrang
//
// Rueckgabewert: true, wenn ein Stoppcodon in der angegebenen Lererichtung
//                gefunden wurde sonst false.
//
// Bemerkung: Positionsangabe ist NICHT C-like. Daher ist stets eine Position 
//            abzuziehen

bool DBCombined::isEnd(long end, 
		   bool reverse) {
    if (reverse) {
	if (end > 3)
	    return (seq[end - 2] == 'a' &&
		    ((seq[end - 3] == 't' &&
		      (seq[end - 4] == 't' || seq[end - 4] == 'c')) ||
		     (seq[end - 3] == 'c' && seq[end - 4] =='t')));
	else
	    return false;
    }
    if (end < (long)strlen(seq) - 2)
	return (seq[end] == 't' &&
		((seq[end + 1] == 'a' &&
		  (seq[end + 2] == 'g' || seq[end + 2] == 'a')) ||
		 (seq[end + 1] == 'g' && seq[end + 2] == 'a')));
    else
	return false;
} // isEnd

//---------------------------------------------------------------------------//

// Private - Methode: setFrames
//
// Beschreibung: Setzt die Leserahmen der Teiltreffer.
//               Der Leserahmen haengt von der Laenge und des Leserahmens
//               des vorherigen (in Leserichtung) Teiltreffers ab. Der
//               erste Teiltreffer besitzt den Leserahmen 0.
//
// Parameter: 'm' - Liste der Treffer, deren Teiltreffer bearbeitet werden

void DBCombined::setFrames(DBMatch_t *m) {
    if (!m || !m->parts)
	return;

    // Alle Treffer werden bearbeitet
    for (DBMatch_t *t = m; t; t = t->next) {

	// Unterschiedliche Vorgehensweisen bie unterschiedlicher Orientierung
	// der Treffer.
	if (t->trend < 0) {

	    // Treffer auf dem Gegenstrang.
	    // Die Teiltreffer muessen "von hinten nach vorne" bewertet werden.
	    // Dazu werden zuerst alle Teiltreffer ausser dem "ersten" in der
	    // Liste (von der Leserichtung her der "letzte")bewertet. 
	    // Daraus ergibt sich der Leserahmen des "ersten" Teiltreffers und 
	    // die anderen Leserahmen lassen sich daraus bestimmen.
	    int lastframe = 0;
	    for (DBParts_t *p = t->parts->next; p; p = p->next) {
		lastframe += (3 - (p->qry_to - p->qry_from + 1) % 3) % 3;
	    }
	    lastframe %= 3;
	    t->parts->frame = lastframe;
	    for (DBParts_t *p = t->parts->next; p; p = p->next) {
		p->frame =  (p->qry_to - p->qry_from + 1 + lastframe) % 3;
		lastframe = p->frame;
	    }
	} else {

	    // Treffer auf dem Vorwaertsstrang.
	    // Daraus ergibt sich eine einfache Vorgehensweise.
	    int nextframe = 0;
	    for (DBParts_t *p = t->parts; p; p = p->next) {
		p->frame = nextframe;
		nextframe = (3 -
			     (p->qry_to - p->qry_from + 1 - p->frame)% 3) % 3;
	    }
  	}
    }
} // setFrames

//---------------------------------------------------------------------------//

// Private - Methode: readProps
//
// Beschreibung: Fuer die Klasse relevante Optionen werden der 
//               Konfigurationsdatei (durch die Klasse DBProperties) entnommen.
//               Liegen keine Angaben ueber die einzelnen Variablen vor, 
//               so werden Default-Werte gesetzt.

void DBCombined::readProps(void) {

    // Filterung nach dem Bezeichner durch diese Variablen moeglich
    try {
        s_ignore = props.getStringArray("prot_ignore_string");
        s_has = props.getStringArray("prot_has_string");
    } catch (DBPropertiesError &err) {
	cerr << "Fehler beim Lesen von "
	     << "\"prot_ignore/has_string\""
	     << "\nWeitere Bearbeitung ohne \"cluster_buckets\"\n";
	s_has = s_ignore = 0;
    }

    // Scoregrenze, die ein Proteinteiltreffer ueberschreiten muss
    try {
        threshold = props.getInt("prot_threshold");
    } catch (DBPropertiesError &err) {
        threshold = 0;
    }

    // Verhaeltnis von Uebereinstimmungen zu Laenge, die ein Proteinteiltreffer
    // ueberschreiten muss
    try {
        ratio = props.getDouble("prot_ratio");
    } catch (DBPropertiesError &err) {
        ratio = 0.8;
    }
    if (ratio < 0.0 || ratio > 1.0) {
	cerr << "Unguelitiger Wert von \"est_ratio\". Gueltige Werte liegen"
	     << "zischen 0 und 1.\n \"est_ratio\" wird auf 0.0 gesetzt, d. h."
	     << " alle Treffer werden berachtet\n"; 
        ratio = 0.0;
    }
} // readProps


//---------------------------------------------------------------------------//

// Private - Methode: namecheck
//
// Beschreibung: Ueberprueft den Bezeichner auf bestimmt Worte.
//              
// Parameter: 'name' - Der zu untersuchende Bezeichner
//
// Rueckgabewert: false - Wenn in dem Bezeichner eines der Worte aus 's_ignore'
//                        oder nicht alle Worte aus 's_has' gefunden wurden.
//                true  - sonst

bool DBCombined::namecheck(const char *name){
    if (s_ignore)
        for (int i = 0; s_ignore[i] != PROP_STRING_END; i++)
            if (strstr(name, s_ignore[i].c_str()) != NULL)
                return false;
    if (s_has)
        for (int i = 0; s_has[i] != PROP_STRING_END; i++)
            if (strstr(name, s_has[i].c_str()) == NULL)
                return false;
    return true;
} // namecheck

