/*****************************************************************************
 ** Dateiname:    DBMatch.hh
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Bietet Strukturen, die intern die Datenbanktreffer verwalten
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 28.9.2002  | Oliver Schoeffmann      | Erzeugung der Datei
 ** 12.1.2003  | Oliver Schoeffmann      | Hinzufuegen der copy-Funktionen
 ** 12.4.2003  | Oliver Schoeffmann      | query, sbjct und hitline ergaenzt
 ** 23.04.2004 | Mario Stanke            | Ausgabemethoden von DBMatch_t und DBParts_t
 *****************************************************************************/

#ifndef __db_match_hh__
#define __db_match_hh__

#include <string>
#include <iostream>

#include <DBError.hh>
using namespace std;

//
// Klasse DBMatchError
//
// Fehlerklasse von DBMatch_t, abgeleitet von DBError
// Die Art des Fehlers wird durch "m" mitgeteilt

class DBMatchError : public DBError {
 public:
    DBMatchError(string m) : DBError(m) {}
};


// Struktur: DBParts_t
//
// Bietet die Moeglichkeit einen Teiltreffer aus den Suchergebnissen 
// zu speichern. Des weiteren wird die Struktur dafuer benutzt und einen 
// Teil eines Eregebniskomplexes zu speichern.

struct DBParts_t {

    // Konstruktor

    DBParts_t();

    // Destruktor

    ~DBParts_t();

    // Methode: copy
    //
    // Beschreibung: Erstellt eine Kopie dieser Struktur (ohne Nachfolger)
    // 
    // Exception: Speicherfehler

    DBParts_t *copy() throw (DBMatchError);

    // Methode: copy_all
    //
    // Beschreibung: Erstellt eine Kopie der Liste ab der aktuelle Struktur
    //
    // Exception: Speicherfehler

    DBParts_t *copy_all() throw (DBMatchError);

    // Methode: swap
    //
    // Beschreibung: Invertiert die Stuktur. qry_from und qry_to werden 
    //               vertauscht. Ebenso sbj_from und sbj_to. 
    //               Die Zeilen mit den Sequenzen und die Zeile mit den 
    //               Uebereinstimmungen werden umgedreht.

    void swap();
    
    // Variablen:
    // Ausdehnung des Teiltreffers in der Eingabe- (qry) und 
    // Datenbanksequenz (sbj)
    long qry_from, qry_to, sbj_from, sbj_to;

    // Laenge, Score, Anzahl an Uebereinstimmungen bzw. positiv
    // bewertete Positionen
    long length, score, akt_id, akt_pos;

    // Leserahmen
    int frame;
    
    // e-Wert
    double e_value;

    // Markierungen, ob ein Startcodon, Stoppcodon oder Splicesites
    // gefunden wurden (in den Komplexen)
    bool start, end, ass, dss;

    // Naechster Teiltreffer oder Teil des Komplexes
    DBParts_t *next;

    // Genaue Sequenzabschnitte der Eingabe- und Datenbanksequenz sowie
    // die positionsweise Uebereinstimmungen zwischen den beiden Squenzen
    string query, subject, hitline;

}; // struct DBParts_t

// Methode: operator<< 
//
// Beschreibung: Gebe wichtige Daten des DBParts_t aus.

ostream& operator<< ( ostream& out, DBParts_t& part);

// Struktur: DBMatch_t
//
// Bietet die Moeglichkeit einen Treffer inkl. Teiltreffer bzw. einen Komplex
// mit allen Teilen zu speichern.

struct DBMatch_t {

    // Konstruktor

    DBMatch_t();

    // Destruktor

    ~DBMatch_t();

    // Methode: copy
    //
    // Beschreibung: Erstellt eine Kopie dieser Struktur (ohne Nachfolger)
    //               mit allen Teilen
    //
    // Exception: Speicherfehler

    DBMatch_t *copy() throw (DBMatchError);


    // Methode: copy
    //
    // Beschreibung: Erstellt eine Kopie der Liste ab der aktuelle Struktur
    //               mit den zugehoerigen Teilen
    //
    // Exception: Speicherfehler

    DBMatch_t *copy_all() throw (DBMatchError);
    
    // Variablen:
    // Bezeichner
    char *name;

    // Gesammtscore aller Eintraege und Laenge des Datenbankeintrags
    long wholeScore, wholeLength;

    // Orientierung des Treffers
    int trend;

    // Kleinster e-Wert der Teiltreffer bzw. zugehoerige Clustergruppe
    double cluster;

    // Naechster Treffer oder Komplex
    DBMatch_t *next;

    // Teiltreffer bzw. Teile des Komplexes
    DBParts_t *parts;

}; // struct DBMatch_t

// Methode: operator<< 
//
// Beschreibung: Gebe wichtige Daten des DBMatch_t aus.

ostream& operator<< ( ostream& out, DBMatch_t& match);


#endif
