/*****************************************************************************
 ** Dateiname:    DBProt.hh
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Bietet die Proteinverarbeitung an
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 18.2.2003  | Oliver Schoeffmann      | Erzeugung der Datei
 *****************************************************************************/

#ifndef __dbprot_hh__
#define __dbprot_hh__

#include <string>

#include <DBError.hh>
#include <DBProperties.hh>

using namespace std;

//
// Klasse DBProtError
//
// Fehlerklasse von DBProt, abgeleitet von DBError
// Die Art des Fehlers wird durch "m" mitgeteilt

class DBProtError : public DBError {

public:
    DBProtError(const string &m) : DBError(m) {}

}; // class DBProtError


// Klasse DBProt
//
// Stellt Methoden fuer die Proteinverarbeitung bereit

class DBProt {
public:

    // Konstruktor
    //
    // Parameter: ein Objekt vom Typ 'DBProperties'

    DBProt(DBProperties properties);

    // Konstruktor

    DBProt();

    // Destruktor

    ~DBProt();

    // Methode: pureProcess
    //
    // Beschreibung: Fuert die Protein-Verarbeitung durch
    //
    // Parameter: 'sequence' - Die zu untersuchende Sequenz
    //            'Protfile' - Eine Datei, in der schon die Suchergebnisse
    //                         einer Protein-Suche enthalten sind.
    //                         Default: 0 (Protein-Suche selbst durchfuehren)
    //            'cluster'  - true: Protein-Treffer werden in verschiedene 
    //                               Gruppen (nach e-Wert) einsortiert.
    //                         false: alle Protein-Treffer werden gemeinsam 
    //                                betrachtet
    //                         Default: false.
    //
    // Rueckgabewert: Eine Liste der Ergebnisse der Protein-Verarbeitung
    // 
    // Exception: Speicherfehler, Fehler von "ausserhalb"

    DBMatch_t *pureProcess(const char *sequence,
			   const char *protfile = 0,
			   const bool cluster = false) throw (DBProtError);


    // Methode: setProperties
    //
    // Beschreibung: Weist der Klasse ein (neues) DBProperties Objekt zu
    //
    // Parameter: Ein Objekt vom Typ 'DBProperties'

    void setProperties(DBProperties properties);

private:

    // Private - Methoden
    //
    // Beschreibung: s. Implementationsdatei

    DBMatch_t *cleanup(DBMatch_t *m) throw (DBProtError);

    void clustern(DBMatch_t *m);

    void smooth(DBMatch_t *m);

    void includedIntrons(DBMatch_t *m) throw (DBProtError);

    DBMatch_t *findComplex(DBMatch_t *m) throw (DBProtError);

    bool hasIntron(DBParts_t *p, 
		   DBParts_t *q, 
		   int trend);

    void rateHits(DBMatch_t *m);


    void frontcut(DBParts_t *q,
		  int &diff);

    void backcut(DBParts_t *p, 
		 int &diff);

    bool isStart(long start, 
		 bool reverse = false);

    bool isEnd(long end, 
	       bool reverse = false);

    void setFrames(DBMatch_t *m);

    void readProps(void);

    bool namecheck(const char *name);

private:

    // Private - Variablen

    const char *seq;
    string *s_ignore, *s_has;
    long threshold;
    int cutoff;
    double ratio, *cluster_buckets;
    DBProperties props;
}; // class DBProt

#endif
