/*****************************************************************************
 ** Dateiname:    DBStore.cc
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Klasse zur Speicherung von Treffern
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 12.12.2002 | Oliver Schoeffmann      | Erzeugung der Datei
 ** 20.12.2002 | Oliver Schoeffmann      | Funktion zum mischen der EST
 ** 20.4.2003  | Oliver Schoeffmann      | Vergleich der "Traeger"
 *****************************************************************************/

#include <DBStore.hh>
using namespace std;

///////////////////////////////////////////////////////////////////////////////
// Public - Methoden
///////////////////////////////////////////////////////////////////////////////

// Konstruktor
//
// Einige Variablen benoetigen Startwerte

DBStore::DBStore() { 
    match = end = 0; 
    size = 0; 
}

//---------------------------------------------------------------------------//

// Destruktor
//
// Die Liste wird wieder freigegeben

DBStore::~DBStore() { 
    delete match; 
}

//---------------------------------------------------------------------------//

// Methode: getMatches
//
// Beschreibung: s. Headerdatei

DBMatch_t *DBStore::getMatches() throw(DBStoreError) {
    DBMatch_t *ret = 0;
    if (size)
	
	// Eine Kopie der Komplexe wird zurueckgeliefert
	try {
	    ret = match->copy_all();
	} catch (DBMatchError &err) {
	    throw DBStoreError((string)"DBScore::getMatches:"
			       " Fehler verursacht durch:\n"
			       + err.getMessage());
	}
    return ret;
} // getMatches

//---------------------------------------------------------------------------//

// Methode: clear
//
// Beschreibung: s. Headerdatei

void DBStore::clear() {
    delete match;
    match = end = 0;
    size = 0;
} // clear

//---------------------------------------------------------------------------//

// Methode: add
//
// Beschreibung: s. Headerdatei

void DBStore::add(DBMatch_t *toAdd,
		  bool cluster, 
		  bool single) throw (DBStoreError) {
    if (!toAdd || !toAdd->parts)
	return;
    DBMatch_t *insert;

    // Eine Kopie word erstellt
    try {
	insert = toAdd->copy();
    } catch (DBMatchError &err) {
	throw DBStoreError((string) "DBStore::add:\n" 
			   "Fehler verursacht durch:\n" + err.getMessage());
    }

    // Der erste Komplex
    if (!size) {
	match = end = insert;
	size++;
	return;
    }

    // single == true bedeutet keinen Vergleich, 
    // alle Komplexe werden gespeichert
    if (single) {
	size++;
	end->next = insert;
	end = end->next;
	return;
    }

    // Unterschied, ob neuer Komplex aus einem oder mehreren Teilen besteht
    if (insert->parts->next == 0) {
	newSingle(insert);
    } else {
	newComplex(insert);
    }
    return;
}

//---------------------------------------------------------------------------//

// Methode: cutWith
//
// Beschreibung: s. Headerdatei

bool DBStore::cutWith(DBMatch_t *bm, 
		      bool cluster){

    // Verglichen werden die "Traeger" der Komplexe.
    // Das ist der Abschnitt aus der Eingabesequenz, zwischen 
    // qry_from des ersten Teiltreffers und
    // qry_to des letzten Teiltreffers
    long tto = 0, mto = 0, mfrom = bm->parts->qry_from, tfrom = 0;
    for (DBParts_t *p = bm->parts; p; p = p->next)
	mto = p->qry_to;
    for (DBMatch_t *t = match; t; t = t->next) {

	// Gleiche Orientierung und evtl. Clustergruppe werde fuer einen 
	// Vergleich vorausgesetzt
	if ((t->trend != bm->trend) || 
	    (cluster && t->cluster != bm->cluster))
	    continue;
	tfrom = t->parts->qry_from;
	for (DBParts_t *p = t->parts; p; p = p->next)
	    tto = p->qry_to;
	if (mfrom < tto && tfrom < mto)
	    return true;
    }

    // kein Schnitt
    return false;
}

//---------------------------------------------------------------------------//

// Methode: getSize
//
// Beschreibung: s. Headerdatei

long DBStore::getSize() {
    return size;
}

//---------------------------------------------------------------------------//

// Methode: cleanup
//
// Beschreibung: s. Headerdatei

void DBStore::cleanup() {
    if (!size)
	return;
    DBMatch_t *tmp = match;
    long tmpsize = size;    
    size = 0;
    try {
	for (DBMatch_t *t = tmp; t; t = t->next)
	    this->add(t);
    } catch (DBStoreError &err) {
	match = tmp;
	size = tmpsize;
    }
}

///////////////////////////////////////////////////////////////////////////////
// Private - Methoden
///////////////////////////////////////////////////////////////////////////////

// Bemerkung:
// Die folgenden Methoden dienen dem Einsortieren von EST-Komplexen.
// Bei Protein-Komplexen reicht die Angabe von 'cutWith'


// Private - Methode: singleCut
//
// Beschreibung: Ein Komplex aus einem Teil (single) wird mit einem anderen
//               Komplex (beliebig viele Teile) verglichen.
//               Es wird versucht die Komplexe zusammenzulegen.
//               Der 'single' wird als "exonpart" angesehen, als besitzt er
//               eine relativ geringe Aussagekraft. Ragt er nun bei dem anderen
//               Komplex in ein Intron, so wird dies nicht als Fehler angesehen
//
// Parameter: 'single' - Komplex aus einem Teil
//            'other'  - Komplex mit dem verglichen werden soll
//
// Rueckgabewert: 'true'  - es liegt ein Schnitt vor
//                'false' - es liegt kein Schnitt vor
//
// Nebeneffekt: An den "Raendern" von 'other' werden evtl. Veraenderungen
//              vorgenommen, die aus dem Zusammenlegen beider Komplexe
//              resultieren.

bool DBStore::singleCut(DBMatch_t *single, 
			DBMatch_t *other) {
    DBParts_t *mp = other->parts;
    long from = mp->qry_from;
    while(mp->next)
	mp = mp->next;
    
    // Schnitt der Traeger ermitteln
    if (from <= single->parts->qry_to &&
	single->parts->qry_from <= mp->qry_to) {
	
	// Evtl. die Raender von 'other' anpassen
	if (from > single->parts->qry_from)
	    other->parts->qry_from = single->parts->qry_from;
	
	if (mp->qry_to < single->parts->qry_to)
	    mp->qry_to = single->parts->qry_to;
	
	return true;
    }
    return false;
} // singleCut

//---------------------------------------------------------------------------//

// Private - Methode: newSingle
//
// Beschreibung: Ein Komplex aus nur einem Teil wird mit den bisherigen 
//               Komplexen verglichen und evtl zusammengelegt
//
// Parameter: 'insert' - Der neue Komplex

void DBStore::newSingle(DBMatch_t *insert) {
    bool merged = false;

    // Vergleich mit allen bisherigen Komplexen
    for (DBMatch_t *m = match; m; m = m->next) {
	merged = singleCut(insert, m) || merged;
    }

    // Existierte kein Schnitt mit einem der bisherigen Komplexe,
    // so wird der neue Komplex hinzugefuegt und ansonsten geloescht.
    if (!merged) {
	end->next = insert;
	end = end->next;
	size++;	
    } else {
	delete insert;
    }

} // newSingle

//---------------------------------------------------------------------------//

// Private - Methode: newComplex
//
// Beschreibung: Ein neuer Komplex aus mehreren Teilen wird mit den 
//               bisherigen Komplexen verglichen und evtl. zusammengelegt
//
// Parameter: 'insert' - Der neue Komplex
//
// Exception: Kopierfehler

void DBStore::newComplex(DBMatch_t *insert) throw (DBStoreError) {
    long ifrom = insert->parts->qry_from, ito = 0;

    // Der Traeger des neuen Komplexes wird bestimmt
    for (DBParts_t *p = insert->parts; p; p = p->next)
	ito = p->qry_to;
    bool merged = false;

    // Es werden alle bisherigen Komplexe verglichen
    for (DBMatch_t *m = match; m; m = m->next) {
	if (m->parts->next == 0) {

	    // Der vorhandene Komplex besteht nur aus einem Teil.
	    if (singleCut(m, insert)) {
		try {
		    delete m->parts;
		    m->parts = insert->parts->copy_all();
		    m->trend = insert->trend;
		    merged = true;
		} catch (DBMatchError &err) {
		    throw DBStoreError((string) "DBStore::newComplex:" 
				       " Fehler erzeugt durch:\n" 
				       + err.getMessage());
		}
	    } 
	} else {

	    // Der vorhandene Komplex besteht auch aus mehreren Teilen
	    if (m->trend != insert->trend)
		continue;
	    long mfrom = m->parts->qry_from, mto = 0;
	    for (DBParts_t *p = m->parts; p; p = p->next)
		mto = p->qry_to;

	    // Liegt ein Schnitt der Traeger vor? Nein => weiter
	    if (!(mfrom <= ito && ifrom <= mto))
		continue;
		
	    // Vor allen die "Introns" in der Teiltreffern werden 
	    // aus Widersprueche untersucht.
	    DBParts_t *mp = m->parts, *ip = insert->parts;
	    bool old_first = true;

	    // Die Pisition des Schnittes wird lokalisiert und
	    // die Komplexe evtl. schon zusammengelegt
	    if (mp->qry_from <= ip->qry_from) {
		while (mp->next && mp->qry_to != ip->qry_to) {
		    mp = mp->next;
		}
		if (!mp->next) {
		    // erster Teil von insert und letzter Teil von match
		    // ueberschneiden sich => ein Komplex

		    if (mp->qry_from < ip->qry_to &&
			ip->qry_from < mp->qry_to) {
			mp->qry_to = ip->qry_to;
			try {
			    mp->next = ip->next->copy_all();
			} catch (DBMatchError &err) {
			    throw DBStoreError((string) "DBStore::newComplex" 
					       " Fehler erzeugt durch:\n"
					       + err.getMessage());
			}
			merged = true;
		    }
		    continue;
		}
	    } else {
		old_first = false;
		while (ip->next && ip->qry_to != mp->qry_to) {
		    ip = ip->next;
		}
		if (!ip->next) {

		    if (ip->qry_from < mp->qry_to &&
			mp->qry_from < ip->qry_to) {
			ip->qry_to = mp->qry_to;
			try {
			    ip->next = mp->next->copy_all();
			} catch (DBMatchError &err) {
			    throw DBStoreError((string) "DBStore::add:\n" 
					       + err.getMessage());
			}
			delete m->parts;
			try {
			    m->parts = insert->parts->copy_all();
			} catch (DBMatchError &err) {
			    throw DBStoreError((string) "DBStore::newComplex:" 
					       " Fehler erzeugt durch:\n"
					       + err.getMessage());
			}
			merged = true;
		    }
		    continue;
		}
	    }

	    // Es wird nach Widerspruechen innerhalb beider Komplexe gesucht.
	    bool contradict = false;
	    while (ip->next && mp->next) {
		if (ip->qry_to != mp->qry_to  ||
		    ip->next->qry_from != mp->next->qry_from)
		    contradict = true;
		ip = ip->next;
		mp = mp->next;
	    }

	    // Die beiden Komplexe widersprechen sich nicht.
	    // Sie werden zusammengelegt.
	    if (!contradict) {
		merged = true;
		if (old_first) {
		    if (ip->next) {
			mp->qry_to = ip->qry_to;
			try {
			    mp->next = ip->next->copy_all();
			} catch (DBMatchError &err) {
			    throw DBStoreError((string) "DBStore::newComplex:" 
					       " Fehler erzeugt durch:\n"
					       + err.getMessage());
			}
		    } else {
			if (!mp->next && mp->qry_to < ip->qry_to)
			    mp->qry_to = ip->qry_to;
		    }
		} else {
		    if (mp->next) {
			ip->qry_to = mp->qry_to;
			try {
			    ip->next = mp->next->copy_all();
			} catch (DBMatchError &err) {
			    throw DBStoreError((string) "DBStore::newComplex:" 
					       " Fehler erzeugt durch:\n"
					       + err.getMessage());
			}
		    } else {
			if (!ip->next && ip->qry_to < mp->qry_to)
			    ip->qry_to = mp->qry_to;
		    }
		    delete m->parts;
		    try {
			m->parts = insert->parts->copy_all();
		    } catch (DBMatchError &err) {
			throw DBStoreError((string) "DBStore::newComplex:" 
					   " Fehler erzeugt durch:\n"
					   + err.getMessage());
		    }
		}
	    } // if (!contradict)
	}
    } // for

    // Liegt kein Schnitt mit den bisherigen Komplexen vor,
    // so wird der neue Komplex hinzugefuegt.
    if (!merged) {
	end->next = insert;
	end = end->next;
	size++;	
    } else {
	delete insert;
    }

} // newComplex


