/*****************************************************************************
 ** Dateiname:    DBBlast.cc
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Die Klasse DBBlast stellt die Moeglichkeiten fuer die 
 **               Datenbankanfragen und das parsen der Ausgabe zur Verfuegung
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 2.11.2002  | Oliver Schoeffmann      | Erzeugung der Datei
 ** 9.11.2002  | Oliver Schoeffmann      | Sortierung der Teiltreffer
 ** 12.4.2003  | Oliver Schoeffmann      | Extrahieren der einzelnen 
 **            |                         | Trefferzeilen (Sequenzen)
 ** 16.4.2004  | Stefanie Dahms          | Erweiterung fuer wublast  
 ** 30.12.2005 | Mario Stanke            | BLAT
 *****************************************************************************/

#include <cstdlib>
#include <fstream>
#include <string>
#include <cstdio>
#include <iostream>

#include <DBBlast.hh>
using namespace std;


///////////////////////////////////////////////////////////////////////////////
// Public - Methoden
///////////////////////////////////////////////////////////////////////////////

// Konstruktor
//
// Uebernimmt das DBProperties Objekt.
// 'file' markiert, ob das Suchergebnis geloecht werden muss.

DBBlast::DBBlast(DBProperties properties) {
    prop = properties;
    file = true;
}

//---------------------------------------------------------------------------//

// Destruktor
//
// Schliesst die evtl. noch offene Datei mit den Suchergebnissen

DBBlast::~DBBlast() {
    ifstrm.close();
}

//---------------------------------------------------------------------------//

// Methode: process (2 Parameter)
//
// Beschreibung: s. Headerdatei
//               Ruft die Methode 'process' (3 Parameter) auf und veranlasst, 
//               dass die Suchergebnisse geloescht werden (file = false)

DBMatch_t *DBBlast::process(const char *sequence,
			    DB_t db) throw(DBBlastError){
    file = false;
   DBMatch_t *ret = process(sequence, "blast_db_output_tmp", db);
    file = true;
    return ret;
} // process (2 Parameter)


//---------------------------------------------------------------------------//

//
// Methode: process (3 Parameter)
//
// Beschreibung: s. Headerdatei

DBMatch_t *DBBlast::process(const char *sequence, 
			    const char *filename,
			    DB_t db) throw(DBBlastError){
    string  word;
    int blasttype = 0; // 0: ncbi-BLAST, 1: wublast: 2: blat
    string blastcmd("");
    try{
	word = prop.getString("blast_programm");
	if(strstr(word.c_str(),"wublast") != NULL){
	    blasttype = 1;
	} else if ( strstr(word.c_str(),"blat") != NULL){
	    blasttype = 2;
	} else {
	    blasttype = 0;
	    blastcmd += "blastall";
	}
    } catch(DBPropertiesError &err) {
	blasttype = 0;
	blastcmd += "blastall";
    }
    // Die der Suchanfrage entsprechenden Optionen werden aus der
    // Konfigurationsdatei entnommen.
    if (db == PROT) {
	try {
	    word = prop.getString("prot_programm");
	    if(blasttype == 0)
	      blastcmd += " -p " + word;
	    else
	      blastcmd += word;
	} catch (DBPropertiesError &err) {
	    if(blasttype == 0)
		blastcmd += " -p blastx ";
	    else if (blasttype == 1) 
		blastcmd += " blastx ";
	    else 
		blastcmd += " blat ";
	}
	try {
	    word = prop.getString("prot_db");
	    if(blasttype == 0)
		blastcmd += " -d " + word;
	    else
		blastcmd += word;
	} catch (DBPropertiesError &err) {
       
	}
	try {
	    word = prop.getString("prot_matrix");
	    blastcmd += " -M " + word;
	} catch (DBPropertiesError &err) {
	}
	try {
	    word = prop.getString("prot_gap_open");
	    blastcmd += " -G " + word;
	} catch (DBPropertiesError &err) {}
	try {
	    word = prop.getString("prot_gap_extend");
	    blastcmd += " -E " + word;
	} catch (DBPropertiesError &err) {
	}
    } 
    else {
      try {
	word = prop.getString("est_programm");
	if(blasttype == 0)
	    blastcmd += " -p " + word;
	else
	    blastcmd += word;
      } catch (DBPropertiesError &err) {
	  if(blasttype == 0)
	      blastcmd += " -p blastn ";
	  else if (blasttype == 1)
	      blastcmd += " blastn ";
	  else 
	      blastcmd += " blat ";
      }
      try {
	word = prop.getString("est_db");
	if(blasttype == 0)
	    blastcmd += " -d " + word;
	else
	    blastcmd+= word;
      } catch (DBPropertiesError &err) {}
      try {
	word = prop.getString("est_match");
	blastcmd += " -r " + word;
      } catch (DBPropertiesError &err) {
      }
      try {
	word = prop.getString("est_mismatch");
	blastcmd += " -q " + word;
      } catch (DBPropertiesError &err) {
      }
      try {
	word = prop.getString("est_gap_open");
	blastcmd += " -G " + word;
      } catch (DBPropertiesError &err) {
      }
      try {
	word = prop.getString("est_gap_extend");
	blastcmd += " -E " + word;
      } catch (DBPropertiesError &err) {
      }
    }
    
    char *seq_file = "blast_db_seq_tmp";
    if (blasttype == 0)
	blastcmd = blastcmd + " -i " + seq_file + " -o " + filename;
    else if (blasttype == 1)
	blastcmd = blastcmd + " "+ seq_file + " -o " + filename;
    else 
	blastcmd = blastcmd + " "+ seq_file + " -out=wublast " + filename;	
    try {
	if (blasttype == 1){
	    word = prop.getString("wublast_options");
	    blastcmd = blastcmd + " " + word;
	    if (db == PROT) 
		word = prop.getString("wublastx_options");
	    else 
		word = prop.getString("wublastn_options");
	    blastcmd = blastcmd + " " + word;
	} else if (blasttype == 0){
	    word = prop.getString("blast_options");
	    blastcmd = blastcmd + " " + word;
	}
    } catch (DBPropertiesError &err) {
    }
    
    // Die Sequenz wird zur Suchanfrage in eine Datei geschrieben.
    ofstream ofstrm(seq_file);
    if (ofstrm.fail())
	throw DBBlastError((string)"DBBlast|process: " + seq_file + 
			   " Fehler bei der Datenbanksuche (Dateierstellung)");
    ofstrm << ">tempseq\n" << sequence << endl;
    ofstrm.close();
    
    // Die Suchanfrage wird durchgefuehrt
    cout << "running " << blastcmd << endl;
    system(blastcmd.c_str());
    
    // Die Suchergebnisse werden ausgewertet
    DBMatch_t *ret = parseFile(filename);
    blastcmd.erase();
    blastcmd = blastcmd + "rm -rf " + seq_file;

    // Die Datei mit den Suchergebnissen wird evtl. geloescht.
    if (!file) {
	blastcmd = blastcmd + " " + filename;
    }
    system(blastcmd.c_str());
    // possible source of errors. after reopening the file the failbit is set
    // Maybe wublast does not close the file?
    // For a temp. bugfix see method parseFile below

    return ret;

} // process (3 Parameter)


//---------------------------------------------------------------------------//

// Methode: parseFile
// 
// Beschreibung: s. Headerdatei

DBMatch_t *DBBlast::parseFile(const char *filename) throw(DBBlastError){
  try {
    // Oeffnen der Datei
    if (!filename)
      throw DBBlastError("DBBlast|parseFile: "
			 "keinen Dateinamen angegeben");
    if (ifstrm.is_open())
	ifstrm.close();
    ifstrm.open(filename);
    
    if (ifstrm.fail()){ // probably the fault of wublast
      ifstrm.clear();
      if (ifstrm.is_open())
	ifstrm.close();
      ifstrm.open(filename);
    }
    
    if (ifstrm.fail()){
      throw DBBlastError((string) "DBBlast|parseFile: " + filename + 
			 " ist nicht lesbar");
    }
   
    // Ueberpruefen des Dateiheaders
    string word, line; 
    if (!(ifstrm >> word)) {
	return 0; // empty file, no matches if searched with blat
    }
    if (strstr(word.c_str(),"BLAST") == NULL)
      throw DBBlastError((string) "DBBlast|parseFile: " + filename + 
			 " hat fehlerhaften Aufbau");
    //zugefuegt
    bool wublast=false;

    do {
      getline(ifstrm, line);
      if(strstr(line.c_str(),"Gish")!=NULL){	
	wublast=true;
      }
      if(wublast){
	if (strstr(line.c_str(),"NONE")!= NULL)
	  return 0;
      }
      else{
	if (strstr(line.c_str(),"No hits found") != NULL)
	  return 0;
      }
    } while (!ifstrm.eof() && 
	     strstr(line.c_str(),"Sequences producing") == NULL);
    getline(ifstrm, line); // Leerzeile
    
    do {
      getline(ifstrm, line);
      //} while (!ifstrm.eof() && line.find("ALIGNMENTS") == string::npos);
    } while (!ifstrm.eof() && line.length() != 0); 
    
    if (ifstrm.eof())
      return NULL;
    
    //  throw DBBlastError((string)"DBBlast|parseFile: " + filename + " hat fehlerhaften Aufbau");
    
    ///////////////////////////////////////////////////////////////////////
    // Extrahieren der Suchergebnisse
    ///////////////////////////////////////////////////////////////////////
    
    DBMatch_t *ret = 0, *end = 0;
    DBParts_t *akt = 0;
    string name;
    char iname[256], ivalue[32], *ipos;
    do{      
      getline(ifstrm, line);
    }while(!ifstrm.eof() && line[0]!='>');//line.find('>') != string::npos);
    

    // Liste mit "Treffern" erstellen. Zu jedem Treffer gibt es einzelne 
    // Teiltreffer.
    while (!ifstrm.eof() && line.find('>') != string::npos) {
       
      if (ret == 0) {
	ret = new DBMatch_t;
	end = ret;
      } else {
	end->next = new DBMatch_t;
	end = end->next;
      }
      if (!end)
	throw DBBlastError ("DBBlast|parseFile: "
			    "Speicherfehler");
      
      // Laenge und Bezeichner des Datenbankeintrages, das diesen Treffer
      // und die folgenden Teiltreffer verursacht hat
      while ((ipos = strstr(line.c_str(),"Length = ")) == NULL) {
	
	name.append(line);
	name.append("\n");
	getline(ifstrm, line);
      }
      if (!name.length())
	throw DBBlastError("DBBlast|parseFile: "
			   "Kein Bezeichner gefunden");
      end->name = new char[name.length() + 1];
      if (!end->name)
	throw DBBlastError("DBBlast|parseFile: "
			   "Speicherfehler");
      strcpy(end->name, name.c_str());
      name.erase();
      if (2 != sscanf(ipos, "%s = %s", iname, ivalue))
	throw DBBlastError((string)"DBBlast|parseFile: " + 
			   filename + " hat fehlerhaften Aufbau");
      end->wholeLength = atoi(ivalue);
      
      // Extrahieren der zugehoerigen Teiltreffer
      akt = 0;
     
      if(wublast){
	
	while(strstr(line.c_str(),"Strand HSPs")==NULL){
	  getline(ifstrm, line);
	}
      }
      // evtl Zeilen weg
      
      
      getline(ifstrm, line);
      while (true) {
	
	getline(ifstrm, line);
	line += '\0';
	      
	// Neuer Treffer gefunden. Der letzte Teiltreffer wird 
	// hinzugefuegt.
	if (ifstrm.eof() || line[0] == '>') {
	  insert(end, akt);

	  line.erase(line.length() - 1, 1);
	  break;
	}
	
	// Neuen Teiltreffer erstellen und dessen Score auslesen
	if ((ipos = strstr(line.c_str(),"Score = ")) != NULL) {
	  if (akt)
	    insert (end, akt);

	  akt = new DBParts_t;
	  if (!akt)
	    throw DBBlastError("DBBlast|parseFile; "
			       "Speicherfehler");
	  if (2 != sscanf(ipos, "%s = %s", iname, ivalue))
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");
	  
	  akt->score = atoi(ivalue);
	  end->wholeScore += akt->score;
	} // Score

	// E-Wert des Teiltreffers auslesen
	if ((ipos = strstr(line.c_str(),"Expect")) != NULL) {
	  if (2 != sscanf(ipos, "%s = %s", iname, &ivalue[1]))
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");
	  
	  // vor den e-Wert evtl. eine 1 setzen, falls er gleich mit
	  // der Angabe e-x beginnt (fuer atof)
	  ivalue[0] = '0';
	  if (!isdigit(ivalue[1]))
	    ivalue[0] = '1';
	  akt->e_value = atof(ivalue);
	}
	
	// Anzahl der Uebereinstimmungen (zwischen Eingabe- und 
	// Datenbaksequenz) des Teiltreffer auslesen
	if ((ipos = strstr(line.c_str(),"Identities")) != NULL) {
	  if (2 != sscanf(ipos, "%s = %s", iname, ivalue))
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");
	  
	  if ((ipos = strstr(ivalue,"/")) == NULL)
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");
	  
	  akt->akt_id = atoi(ivalue);
	  akt->length = atoi(ipos + 1);
	}
	      
	// Positives: Anzahl an Positionen, die einen Score > 0
	// erzielt habe (nur bei Proteinergebnissen)
	if ((ipos = strstr(line.c_str(),"Positives")) != NULL) {
	  if (2 != sscanf(ipos, "%s = %s", iname, ivalue))
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");
	  
	  akt->akt_pos = atoi(ivalue);
	}
	      
	// Leseraster des Teiltreffers auslesen (Proteine)
	if ((ipos = strstr(line.c_str(),"Frame = ")) != NULL) {
	  // --hinzu--
	  if (2 != sscanf(ipos, "%s = %s", iname, ivalue))
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");
	  
	  akt->frame = atoi(ivalue);
	}
	
	// Orientierung des Teiltreffers (Nukleotiddatenbank)
	if ((ipos = strstr(line.c_str(),"Strand = ")) != NULL) {
	  char cplmt[32];
	  if (3 != sscanf(ipos, "%s = %s / %s", 
			  iname, ivalue, cplmt))
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");

	  akt->frame = (0 == strstr(ivalue, cplmt)) ? 1 : -1;
	}
	
	// Teiltreffersequenzen und Ausdehnung des Teiltreffers
	if (strstr(line.c_str(),"Query: ") != NULL) {
	  long from_tmp;
	  char name_tmp[256], qry_tmp[256], sbj_tmp[256];
	  if (sscanf(line.c_str(), "%s %li %s %li", 
		     name_tmp, &from_tmp, qry_tmp,&akt->qry_to) != 4)
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");
	  
	  if (!akt->qry_from) 
	    akt->qry_from = from_tmp;
	  akt->query += qry_tmp;
	  
	  getline(ifstrm, line); // Fuelllinie
	  getline(ifstrm, line);
	  line += '\0';
	  if (strstr(line.c_str(), "Sbjct: ") == NULL)
	    throw DBBlastError((string)"DBBlast|parseFile: "
			       + filename + 
			       " hat fehlerhaften Aufbau");
	  
	  if (sscanf(line.c_str(), "%s %li %s %li", 
		     name_tmp, &from_tmp, sbj_tmp,&akt->sbj_to) != 4)
	    throw DBBlastError((string)"DBBlast|parseFile: " + 
			       filename + 
			       " hat fehlerhaften Aufbau");
	  if (!akt->sbj_from) 
	    akt->sbj_from = from_tmp;
	  akt->subject += sbj_tmp;
	  
	  
	  // Uebereinstimmungen der Sequenzabschnitte markieren.
	  // Wird fuer die Proteinverarbeitung benoetigt
	  for (int i = 0; qry_tmp[i] != '\0'; i++)
	    akt->hitline += 
	      (qry_tmp[i] == sbj_tmp[i]) || 
	      (qry_tmp[i] == 'X' && isalpha(sbj_tmp[i]))
	      ? '+' : '-';
	  
	} // Query (end)
	
      } // while(true)
      
    } // while (!ifstrm.eof() && line[0] == '>') {
    ifstrm.close();
    /*
    cout << "******* alle Treffer" << endl;
    for (DBMatch_t *treffer = ret; treffer != NULL; treffer=treffer->next) {
      cout <<  "Treffer" <<*treffer << endl;
      }*/

    return ret;
  } // try
  
    // bei auftretenden Fehlern wird die Datei hinterher geschlossen.
  catch (DBBlastError &err) {
    ifstrm.close();
    throw err;
  }
} // parseFile


///////////////////////////////////////////////////////////////////////////////
// Private - Methoden
///////////////////////////////////////////////////////////////////////////////

// Private - Methode: insert
// 
// Beschreibung: Fuegt einem Treffer einen neuen Teiltreffer hinzu.
//               Der neue Teiltreffer wird mit seinem 'Query'-Part nach der 
//               Eingabesequenz ausgerichtet. Er wird nach seinem Startpunkt 
//               des Query-Part in die Liste der Teiltreffer einsortiert.
//               Nur Teiltreffer gleicher Orientierung werden uebernommen
//
// Parameter: 'm' - Treffer dem der neue Teiltreffer hinzugefuegt werden soll
//            'p' - Der neue Teiltreffer
//
// Nebeneffekt: Der Teiltreffer ist hinterher nicht mehr verwendbar. Er ist in 
//              der Liste einsortiert, oder geloescht.
//
// Vorbedingung: keiner der beiden Zeiger ist der 0-Zeiger.

void DBBlast::insert(DBMatch_t *m, 
		     DBParts_t *p) {

    // Ausrichten des Teiltreffers
    if (p->qry_from > p->qry_to)
	p->swap();

    // Erster Teiltreffer legt die Orientierung fest
    if (!m->parts) {
	m->parts = p;
	m->trend = p->frame < 0 ? -1 : 1;
	p->frame = 0;
	m->cluster = p->e_value;
    } else if (m->trend * p->frame > 0) {

	// Nur Teiltreffer mit gleicher Orientierung werden uebernommen
	// und einsortiert ...
	p->frame = 0;
	if (p->e_value < m->cluster)
	    m->cluster = p->e_value;
	if (m->parts->qry_from > p->qry_from) {
	    p->next = m->parts;
	    m->parts = p;
	} else {
	    DBParts_t *t = m->parts;
	    while (t->next && t->next->qry_from < p->qry_from)
		t = t->next;
	    p->next = t->next;
	    t->next = p;
	}
    } else {

	// ... sonst geloescht.
	delete p;
    }
} // insert

