/*****************************************************************************
 ** Dateiname:    DBEst.cc
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: EST-Verarbeitung
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 07.11.2002 | Oliver Schoeffmann      | Erzeugung der Datei
 ** 12.05.2004 | Mario Stanke            | set frame to unknown frame
 *****************************************************************************/

#include <iostream>
#include <unistd.h>

#include <DBBlast.hh>
#include <DBEst.hh>
#include <DBStore.hh>

///////////////////////////////////////////////////////////////////////////////
// Public - Methoden
///////////////////////////////////////////////////////////////////////////////

// Konstruktor
// 
// Beschreibung: s. Headerdatei

DBEst::DBEst(DBProperties properties) {
    setProperties(properties);
}

//---------------------------------------------------------------------------//

// Konstruktor
//
// Beschreibung: Einige Variabeln muessen initialisiert werden. Sonst 
// uebernimmt diese Aufgabe 'setProperties'

DBEst::DBEst() {
    threshold = 0;
    ratio = 0.0;
    cutoff = 30;
    s_has = s_ignore = 0;
}

//---------------------------------------------------------------------------//

// Destruktor
//
// Beschreibung: Gibt den Speicher von zwei Variablen wieder frei

DBEst::~DBEst() {
    delete s_has;
    delete s_ignore;
}

//---------------------------------------------------------------------------//

// Methode: setProperties
//
// Beschreibung: s.Headerdatei. Initialisiert interne Variablen.

void DBEst::setProperties(DBProperties properties) {
    props = properties;
    readProps();
}

//---------------------------------------------------------------------------//
// Methode: proceed
//
// Beschreibung: s. Headerdatei

DBMatch_t *DBEst::proceed(const char *sequence,
			  const char *estfile,
			  const bool single,
			  const bool cluster) throw (DBEstError) {
    if (!sequence)
	throw DBEstError("DBEst::proceed: keine Sequenz angegeben");
    seq = sequence;
    DBBlast blast(props);
    DBMatch_t *ret = 0;

    // Verarbeiten der Suchergebnisse (estfile) oder Durchfuehrung der
    // EST-Datenbankabfrage

    try {
	if (estfile && access(estfile, R_OK)==0) // already have the blast results
	    ret = blast.parseFile(estfile);
	else
	  if (estfile)              // no results yet, but save them once you have them
	    ret = blast.process(sequence, estfile, EST);
	  else                      // dont save results at all
	    ret = blast.process(sequence, EST);

    } catch (DBBlastError &err) {
	throw DBEstError((string)"DBEst::proceed: Fehler produziert durch:\n"
			 + err.getMessage());
    }

    //Gebe alle Matches aus, so wie aus der BLAST-Ausgabe gelesen
    //for (DBMatch_t *match = ret; match != NULL; match = match->next)
    //   cout << *match << endl;

    // Oliver apparently abused the frame data member of DBPart for indicating the strand
    // Therefore: reset all frames to -1 (unknown) as the frame cannot be determined by ESTs
    for (DBMatch_t *match = ret; match != NULL; match = match->next)
      for (DBParts_t *part = match->parts; part != NULL; part=part->next){
	if (part->frame != 0) 
	  cout<< "dachte das koennte nicht sein" << endl;
	part->frame = -1;
      }



    DBStore store;

    // Alle Treffer werden bearbeitet und gemaess der angegebenen Parameter
    // (single, cluster) versucht in die Treffermenge zu integrieren
    for (DBMatch_t *m = ret; m; m = m->next) {
	if (!namecheck(m->name))
	    continue;

	if (cluster)
	    clustern(m);

	try {
	    store.add(cleanup(m), cluster, single);
	} catch (DBStoreError &err) {
	    throw DBEstError((string)"DBEst::proceed: "
			     "Fehler produziert durch:\n"
			     + err.getMessage());
	}
    } // for...

    delete ret;
    try {

	// Die Treffer werden evtl. nochmal gesaeubert (s. DBStore)
	if (!single)
	    store.cleanup();
	ret = store.getMatches();
    } catch (DBStoreError &err) {
	throw DBEstError((string)"DBEst::proceed: Fehler produziert durch:\n"
			 + err.getMessage());
    }
    return ret;
} // proceed


///////////////////////////////////////////////////////////////////////////////
// Private - Methoden
///////////////////////////////////////////////////////////////////////////////

// Private - Methode: cleanup
//
// Beschreibung: Erzeugt aus den Teiltreffern einen Komplex, in dem zwischen 
//               zwei benachbarten Teilen ein Intron gefunden wurde.
// Bemerkung: Es wird nur ein Komplex (mit dem hoechsten erreichten 
//            Gesamtscore) erzeugt.
//
// Parameter: 'bm' - Datenbanktreffer mit den Teiltreffern
//
// Rueckgabewert: Ein Komplex aus den Teiltreffern.
//
// Exceptions: Speicherfehler, Kopierfehler

DBMatch_t *DBEst::cleanup(DBMatch_t *bm) throw (DBEstError) {
    if (!bm || !bm->parts)
        return 0;

    // Wichtige Daten werden mit in das Ergebnis uebernommen
    DBMatch_t *ret = new DBMatch_t;
    if (!ret)
        throw DBEstError("DBEst::cleanup: Speicherfehler");
    ret->cluster = bm->cluster;
    ret->wholeLength = bm->wholeLength;
    ret->name = new char [strlen(bm->name) + 1];
    if (!ret->name)
        throw DBEstError("DBEst::cleanup: Speicherfehler");
    strcpy(ret->name, bm->name);
    
    // Der erste Teiltreffer wird in das Ergebnis uebernommen
    // Die Teiltreffer sind aufsteigend sortiert (nach der Startposition in 
    // der Eingabesequenz). Ein Teiltreffer wird fuer die weiteren 
    // Vorgaenge benoetigt.
    try {
	ret->parts = bm->parts->copy();
    } catch (DBMatchError &err) {
            throw DBEstError((string)"DBEst::cleanup: "
			     "Fehler produziert durch:\n" 
			     + err.getMessage());
    }
    DBParts_t *rp = ret->parts;
    ret->wholeScore = rp->score;
    int restScore = bm->wholeScore - rp->score;
    int introntrend = 0;

    // In den restlichen Teiltreffern wird ein Intron zu dem aktuellen 
    // Teiltreffer gesucht.
    for (DBParts_t *bp = bm->parts ;bp->next; bp = bp->next) {
	if (ratio > ((double) bp->akt_id / bp->length))
	    continue;
	
	// Eine Kopie des zu untersuchenden Teiltreffers wird erstellt
	DBParts_t *bpn;

	try {
	    bpn  = bp->next->copy();
	} catch (DBMatchError &err) {
            throw DBEstError((string)"DBEst::cleanup: "
			     "Fehler produziert durch:\n" 
			     + err.getMessage());
	}

	// Ein Intron wird zwischen dem letzten Teiltreffer des Ergebnisses
	// und dem aktuellen Teiltreffer gesucht.
	introntrend = hasIntron(rp, bpn, ret->trend);


	// Wenn ein Intron gefunden wurde, wird der aktuelle Teiltreffer
	// den Ergebnissen hinzugefuegt.
        if (introntrend) {
            if (!ret->trend)
                ret->trend = introntrend;
	    rp->next = bpn;
	    rp = rp->next;
	    restScore -= bpn->score;
	    ret->wholeScore += bpn->score;
        } else {

	    // Wurde kein Intron gefunden, so wird ueberprueft, ob 
	    // (theoretisch) noch ein groesserer Komplex moeglich ist.
	    // Wenn nicht, dann wird die Suche nach Introns eingestellt.
	    // Wenn doch, dann wird der bisherige Komplex entfernt und
	    // ein neuer Komplex erzeugt.
            if (restScore < ret->wholeScore)
                break;
            delete ret->parts;
            ret->trend = 0;
            ret->parts = bpn;
            rp = ret->parts;
            ret->wholeScore = bpn->score;
            restScore -= bpn->score;
        }
    } // for(;...

    // Der Komplex muss einen gewissen Gesamtscore erreichen (threshold)
    if (ret->wholeScore < threshold) {
        delete ret;
        return 0;
    }


    // An den Enden wird der Komplex um 'cutoff' Positionen gekuerzt.
    // Sind die Abschnitte an den Enden kuerzer als 'cutoff', so werden diese
    // Teile komplett entfernt.
    DBParts_t *pre = 0;
    if (ret->parts->qry_to - ret->parts->qry_from < cutoff) {
	pre = ret->parts;
	ret->parts = ret->parts->next;
	pre->next = 0;
	delete pre;
    } else {
	ret->parts->qry_from += cutoff;
    }
    if (ret->parts) {
	for (pre = 0, rp = ret->parts; rp->next; pre = rp, rp = rp->next);
	if (pre && (rp->qry_to - rp->qry_from < cutoff)) {
	    delete pre->next;
	    pre->next = 0;
	} else if (!pre && (rp->qry_to - rp->qry_from < 3 * cutoff)) {
	    delete ret->parts;
	    ret->parts = 0;
	} else {
	    rp->qry_to -= cutoff;
	}
    }
    if (!ret->parts) {
	delete ret;
	ret = 0;
    }
    return ret;
} // cleanup

//---------------------------------------------------------------------------//

// Private - Methode: clustern
//
// Beschreibung: Der Treffer wird in eine bestimmte Klasse eingeteilt 
//               (nach e-Wert). Das Element 'cluster' des Treffers wird
//               dementsprechend veraendert.
//
// Parameter: 'm' - Der zu bearbeitende Treffer.

void DBEst::clustern(DBMatch_t *m) {
    if (cluster_buckets) {
	for (int i = 0; cluster_buckets[i] != PROP_DOUBLE_END; i++) {
	    if (m->cluster <= cluster_buckets[i]) {
		m->cluster = cluster_buckets[i];
		return;
	    }
	}
	m->cluster = PROP_DOUBLE_END;
    }
}

//---------------------------------------------------------------------------//

// Private - Methode: checkForIntron
//
// Beschreibung: An den uebergebenen Positionen wird der Minimalkonsens fuer
//               ein Intron gesucht. 
//
// Parameter: 'to'   - Endposition des (Vorgaenger-)Exons
//            'from' - Startpisition des (Nachfolger-)Exons
//
// Rueckgabewert: 0  - kein Intron gefunden
//                1  - ein Intron auf dem Vorwaertstrang wurde gefunden
//                -1 - ein Intron auf dem Gegenstrang wurde gefunden.
//
// Bemerkung: Die Positionsangaben sind NICHT C-like.

int DBEst::checkForIntron(long to, 
			  long from) {
    // Introns auf beiden Straengen haben an diesen Positionen das gleiche
    // Nukleotid
    if (seq[to + 1] == 't' && seq[from - 3] == 'a') {

	// Unterschiede an diesen Positionen auf den verschiedenen Straengen
        if (seq[to] == 'g' && seq[from - 2] == 'g')
            return 1;
        if (seq[to] == 'c' && seq[from - 2] == 'c')
            return -1;
    }
    return 0;
}

//---------------------------------------------------------------------------//

// Private - Methode: hasIntron
//
// Beschreibung: Sucht zwischen zwei Teiltreffern ein Intron.
//
// Parameter: 'q'       - "vorderer" Teiltreffer
//            'p'       - "hinterer" Teiltreffer
//            pre_trend - Bisheriger Trend (1, -1) des Komplexes. 
//                        0, wenn keine Vorgabe existiert
//
// Rueckgabewert: 0 - kein Intron gefunden
//                sonst: Trend (1, -1) des gefundenen Introns
//
// Nebeneffekt: Veraendert die "Ausdehnung" der beiden Teiltreffer, wenn
//              ein Intron gefunden wurde.

int DBEst::hasIntron(DBParts_t *p, DBParts_t *q, int pre_trend) {
    int qtrend = q->sbj_from < q->sbj_to ? 1 : 0;
    int ptrend = p->sbj_from < p->sbj_to ? 1 : 0;
    int diff = qtrend * (p->sbj_to - q->sbj_from + 1);

    // In folgenden Faellen kann es kein Intron zwischen den Teiltreffern 
    // geben.
    if (ptrend != qtrend                      ||
	diff < 0                              ||
	diff > (p->qry_to - p->qry_from + 1)  ||
	diff > (q->qry_to - q->qry_from + 1)  ||
	p->qry_to > q->qry_from                 )
	return 0;


    // Suche nach Intron innerhalb des "Schnittes" zwischen den Teiltreffern
    if (diff == 0)
	return checkForIntron (p->qry_to, q->qry_from);
    
    int imark = -1, itrend = 0, itmp = 0, count = 0;
    for (int i = 0; i <= diff; i++) {
	if ((itmp = checkForIntron(p->qry_to - i, q->qry_from + diff - i))) {
	    if (count == 0) {
		if (pre_trend != 0) {
		    if (itmp == pre_trend) {
			itrend = itmp;
			imark = i;
		    }
		} else {
		    itrend = itmp;
		    imark = i;
		}
	    }
	    if (itmp == itrend)
		count++;
	}
    }

    if (count == 0)
	return 0;

    // Anpassen der beiden Teiltreffer, nach der Vorgabe des Introns
    p->qry_to -= imark;
    p->sbj_to -= qtrend * imark;
    q->qry_from += diff - imark;
    q->sbj_from += qtrend * (diff - imark);

    return itrend;
} // hasIntron

//---------------------------------------------------------------------------//

// Private - Methode: readProps
//
// Beschreibung: Fuer die Klasse relevante Optionen werden der 
//               Konfigurationsdatei (durch die Klasse DBProperties) entnommen.
//               Liegen keine Angaben ueber die einzelnen Variablen vor, 
//               so werden Default-Werte gesetzt.

void DBEst::readProps() {

    // Einteilung der Treffer
    try {
	cluster_buckets = props.getDoubleArray("cluster_buckets");
    } catch(DBPropertiesError &err) {
	cerr << "DBEst::readprops: Fehler beim Lesen der \"cluster_buckets\"."
	     << "\nWeitere Bearbeitung ohne \"cluster_buckets\"\n";
	cluster_buckets = 0;
    }
    
    // Filterung nach dem Bezeichner durch diese Variablen moeglich
    try {
        s_ignore = props.getStringArray("est_ignore_string");
        s_has = props.getStringArray("est_has_string");
    } catch (DBPropertiesError &err) {
	cerr << "Fehler beim Lesen von "
	     << "\"prot_ignore/has_string\""
	     << "\nWeitere Bearbeitung ohne \"cluster_buckets\"\n";
        s_has = s_ignore = 0;
    }

    // Scoregrenze, die ein EST-Komplex erreichen muss
    try {
        threshold = props.getInt("est_threshold");
    } catch (DBPropertiesError &err) {
        threshold = 0;
    }

    // Verhaeltnis von Uebereinstimmungen zu Laenge, die ein EST-Teiltreffer
    // ueberschreiten muss
    try {
        ratio = props.getDouble("est_ratio");
    } catch (DBPropertiesError &err) {
        ratio = 0.0;
    }

    // Anzahl der Positionen, die an den Enden der Komplexe entfernt werden
    try {
	cutoff = props.getInt("est_cutoff");
    } catch (DBPropertiesError &err) {
        cutoff = 30;
    }

    // Verhaeltnis von Uebereinstimmungen zu Laenge, die ein EST-Teiltreffer
    // ueberschreiten muss
    if (ratio < 0.0 || ratio > 1.0) {
	cerr << "Unguelitiger Wert von \"est_ratio\". Gueltige Werte liegen"
	     << "zischen 0 und 1.\n \"est_ratio\" wird auf 0.0 gesetzt, d. h."
	     << " alle Treffer werden berachtet\n"; 
        ratio = 0.0;
    }
} // readProps

//---------------------------------------------------------------------------//

// Private - Methode: namecheck
//
// Beschreibung: Ueberprueft den Bezeichner auf bestimmt Worte.
//              
// Parameter: 'name' - Der zu untersuchende Bezeichner
//
// Rueckgabewert: false - Wenn in dem Bezeichner eines der Worte aus 's_ignore'
//                        oder nicht alle Worte aus 's_has' gefunden wurden.
//                true  - sonst

bool DBEst::namecheck(const char *name) {
    if (s_ignore)
	for (int i = 0; s_ignore[i] != PROP_STRING_END; i++)
	    if (strstr(name, s_ignore[i].c_str()) != NULL)
		return false;
    if (s_has)
	for (int i = 0; s_has[i] != PROP_STRING_END; i++)
	    if (strstr(name, s_ignore[i].c_str()) == NULL)
		return false;
    return true;
} // namecheck
