/*****************************************************************************
 ** Dateiname:    DBMain.cc
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Beinhaltet die main-Funktion. Wertet die 
 **               Kommandozeilenparameter aus
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 22.9.2002  | Oliver Schoeffmann      | Erzeugung der Datei
 *****************************************************************************/

#include <iostream>
#include <string>
#include <fstream>
#include <cstring>
#include <cstdlib>
#include <cstdio>

#include <getopt.h>
#include <dirent.h>
#include <DBAccess.hh>
#include <DBProperties.hh>
#include <DBProcess.hh>
#include <DBGff.hh>


// Hilfsfunktion: usage
//
// Gibt eine kurze Gebrauchsanweisung auf die Standardfehlerausgabe aus.
void usage (char *prog, 
	    string errstr) {
    if (errstr.length())
	cerr << errstr << endl << endl;
    cerr << "Benutzung: " << prog << "\n\t[ -c Konfigurationsdatei ] "
	 << "\n\t[ -o Ausgabedatei ] \n\t[ -e BLAST-Ausgabe EST-Suche ] "
	 << "\n\t[ -p BLAST-Ausgabe Proteinsuche ]"
	 << "\n\t[ -w Stckelungsgre der Eingabe]"
	 << "\n\t[ -d berlapp bei der Stckelung]"
	 << "\n\t[ -P ]"
	 << "\n\t[ -E ]\n\t[ -C ]\n\t[ -S ]"
	 << "\n\t-i Eingabedatei\n\n"
	 << "-P: nur Proteinverarbeitung\n"
	 << "-E: nur EST-Verarbeitung\n"
	 << "-C: clustern "
	 << "-S: (EST-)Ergebnisse nicht mischen\n";
    exit(1);
}


int main (int argc, 
	  char *argv[]) {
    if (argc == 1)
	usage(argv[0], "Keine Argumente angegeben");

    // Flags fuer die einzelnen Optionen
    bool has_c = false, has_i = false;
    bool has_o = false, has_e = false;
    bool has_p = false, has_P = false;
    bool has_E = false, has_C = false, has_S = false;
    bool has_n = false;
    bool has_w = false, has_d = false;
    
    // Identifying character of the source (P,C,E)
    char sourceChar='C';

    // Variablen fuer die Optionsparameter
    string config, outfile, infile, est_blast, prot_blast, userseqname;
    int w=0, d=0;
    int argument;
    while (-1 != (argument = getopt(argc, argv, "n:c:e:i:o:p:w:d:PECS"))) {
	switch (argument) {
	    case 'n':
		has_n = true;
		userseqname.assign(optarg);
	    case 'P':
		has_P = true;
		break;
	    case 'E':
		has_E = true;
		break;
	    case 'C':
		has_C = true;
		break;
	    case 'S':
		has_S = true;
		break;
	    case 'e':
		if (has_e)
		    cerr << "Mehrfache Option -e angegeben"
			 << " ignoriere weitere Angabe\n";
		else
		    est_blast.assign(optarg);
		has_e = true;
		break;
	    case 'p':
		if (has_p)
		    cerr << "Mehrfache Option -p angegeben"
			 << " ignoriere weitere Angabe\n";
		else
		    prot_blast.assign(optarg);
		has_p = true;
		break;
	    case 'o':
		if (has_o)
		    cerr << "Mehrfache Option -o angegeben"
			 << " ignoriere weitere Angabe\n";
		else
		    outfile.assign(optarg);
		has_o = true;
		break;
	    case 'i':
		if (has_i)
		    cerr << "Mehrfache Option -i angegeben"
			 << " ignoriere weitere Angabe\n";
		else 
		    infile.assign(optarg);
		// infile
		has_i = true;
		break;
	    case 'c':
		if (has_c)
		    cerr << "Mehrfache Option -c angegeben"
			 << " ignoriere weitere Angabe\n";
		else
		    config.assign(optarg);
		has_c = true;
		break;
	    case 'w':
		if (has_w)
		    cerr << "Mehrfache Option -w angegeben"
			 << " ignoriere weitere Angabe\n";
		else
		    w = atoi(optarg);
		has_w = true;
		break;
	    case 'd':
		if (has_d)
		    cerr << "Mehrfache Option -d angegeben"
			 << " ignoriere weitere Angabe\n";
		else
		    d = atoi(optarg);
		has_d = true;
		break;
	    default:
		usage(argv[0], (string)"Invalid Option");
	} // switch
    } // while (-1 ...
    if (optind < argc) {
	usage(argv[0], (string)"Invalid Option: " + argv[optind]);
    }

    string seqname, sequence;
    DBAccess *dba;
    if (!has_i)
	usage(argv[0], "Keine Eingabedatei angegeben\n");
    try {
	dba = new DBAccess(infile.c_str());
    } catch (DBAccessError &err) {
	cerr << "Fehler der Eingabedatei:\n" << err.getMessage() << endl;
	return 1;
    }
    
    DBGen_t *gen;
    try {
	// loop over the sequences in the multiple fasta file
	while (gen = dba->nextItem()) { 
	    sequence = gen->sequence;
	    seqname = gen->name;
	    delete gen;
    
	    if (has_n)
		seqname.assign(userseqname);
    
	    if (!seqname.length())
		seqname.assign("testseq");
    

	    // check consistency of BLAST input/output files
	    // if the sequence is cut in peaces the output filename
	    // should point to a directory
	    if (has_w && has_e)
		if (!opendir (est_blast.c_str())){
		    cerr << "Bei Stueckelung (-w) muss mit -e ein Verzeichnis angegeben werden.\n" << endl;
		    return 1;
		}
	    if (has_w && has_p)
		if (!opendir (prot_blast.c_str())){
		    cerr << "Bei Stueckelung (-w) muss mit -p ein Verzeichnis angegeben werden.\n" << endl;
		    return 1;
		}

	    // fertig: Sequenz

	    DBProperties props;
	    if (has_c)
		try {
		    props.readFile(config.c_str());
		} catch (DBPropertiesError &err) {
		    cout << "Fehler:\n" << err.getMessage() 
			 << "\n Default-Werte werden benutzt.\n";
		}
    
	    // fertig: Properties

	    DBGff gff;

	    double *cluster_buckets = 0;
	    if (has_C) {
		try {
		    cluster_buckets = props.getDoubleArray("cluster_buckets");
		} catch (DBPropertiesError &err) {
		    cerr << "Fehler beim Lesen der \"cluster_buckets\"."
			 << "\nWeitere Bearbeitung ohne \"cluster_buckets\"\n";
		    has_C = false;
		    cluster_buckets = 0;
		}
	    }

	    // Alle benoetigten Objekte erstellt

	    DBProcess go(props);
	    DBMatch_t *result;

	    ofstream ofstrm;
	    // delete the contents of the output file, if it existed before
	    ofstrm.open(outfile.c_str(), ios::trunc);
	    ofstrm.close();

	    // Die Verarbeitung gemaess der Optionen wird durchgefuehrt
	    // Stueckelungsschleife beginnt
	    if (w <= 0)
		w = sequence.length();
	    bool finished=false;
	    string dies_prot_blast= prot_blast, dies_est_blast=est_blast;

	    for (int offset=0; offset < sequence.length() && !finished; offset += w-d){
		int piecesize = w;
		if (offset + w > sequence.length())
		    piecesize = sequence.length()-offset;
		if (offset + piecesize >= sequence.length())
		    finished = true;
		string chunk = sequence.substr(offset, piecesize);
		cerr << "examining chunk from " << offset+1 << " to " << offset + piecesize << endl;
      
		if (has_w && has_p){
		    char *pfilename = new char[seqname.length()+30];
		    sprintf(pfilename, "%s.prot.%dto%d.blast\0", seqname.c_str(), offset+1, offset+piecesize);
		    dies_prot_blast = prot_blast;
		    if (prot_blast[prot_blast.length()-1] != '/')
			dies_prot_blast.append("/");
		    dies_prot_blast.append(pfilename);
		    //cerr << "schreibe datei " << dies_prot_blast << endl; //TEMP Ausgabe
		    delete pfilename;
		}
		if (has_w && has_e){
		    char *efilename = new char[seqname.length()+30];
		    sprintf(efilename, "%s.est.%dto%d.blast\0", seqname.c_str(), offset+1, offset+piecesize);
		    dies_est_blast = est_blast;
		    if (est_blast[est_blast.length()-1] != '/')
			dies_est_blast.append("/");
		    dies_est_blast.append(efilename);
		    //cerr << "schreibe datei " << dies_est_blast << endl; //TEMP Ausgabe
		    delete efilename;
		} else if (has_e) {
		    dies_est_blast = est_blast; 
		}

		if (has_P) {
		    sourceChar='P';
		    cout << "Fuehre Proteinverarbeitung durch...\n";
		    try {
			if (has_p)
			    result = go.processPROT(chunk.c_str(), dies_prot_blast.c_str(),
						    has_C);
			else
			    result = go.processPROT(chunk.c_str(), 0, has_C);
		    } catch (DBProcessError &err) {
			cout << "Fehler:\n" << err.getMessage() << endl;
			return 1;
		    }
		} else  if (has_E) {
		    sourceChar='E';      
		    cout << "Fuehre EST-Verarbeitung durch...\n";
		    try {
			if (has_e)
			    result = go.processEST(chunk.c_str(), dies_est_blast.c_str(),
						   has_S, has_C);
			else
			    result = go.processEST(chunk.c_str(), 0, has_S, has_C);
		    } catch (DBProcessError &err) {
			cout << "Fehler:\n" << err.getMessage() << endl;
			return 1;
		    }
		} else {
		    sourceChar='C';
		    cout << "Fuehre kombiniert Verarbeitung durch...\n";
		    try {
			if (has_e)
			    result = go.process(chunk.c_str(), dies_est_blast.c_str());
			else
			    result = go.process(chunk.c_str());
		    } catch (DBProcessError &err) {
			cout << "Fehler:\n" << err.getMessage() << endl;
			return 1;
		    }
		}
		// Ausgabe der Ergebnisse
		gff.offset = offset;
		if (has_C && cluster_buckets != 0) {
	    
		    // Geclusterte Ausgabe. Die einzelnen Gruppen werden ausgegeben
		    int i;
		    char output[256];
		    if (outfile.length() == 0)
			outfile.assign("cluster_ausgabe");

		    // Einzelne Cluster werden gruppiert
		    for (i = 0; cluster_buckets[i] != PROP_DOUBLE_END; i++) {
			DBMatch_t *tmp = 0, *pre = 0, *part = 0;
			for (DBMatch_t *m = result; m;) {
			    if (m->cluster == cluster_buckets[i]) {
				if (tmp) {
				    tmp->next = m;
				    tmp = tmp->next;
				} else {
				    part = m;
				    tmp = part;
				}
				if (pre) {
				    pre->next = m->next;
				    m = pre->next;
				} else {
				    m = m->next;
				    result = m;
				}
				tmp->next = 0;
				continue;
			    }
			    pre = m;
			    m = m->next;
			}
			if (part) {
			    if (i) {
				sprintf(output, "%s_%g_to_%g", outfile.c_str(),
					cluster_buckets[i-1], cluster_buckets[i]);
			    } else {
				sprintf(output, "%s_kleiner_%g", outfile.c_str(),
					cluster_buckets[i]);
			    }
			    ofstrm.open(output, ios::app);
			    if (ofstrm.fail()) {
				cerr << "Fehler beim oeffnen von " << output
				     << " Standardausgabe wird benutzt.\n";
				gff.write(part, cout, seqname.c_str(), sourceChar);
			    } else {
				gff.write(part, ofstrm, seqname.c_str(), sourceChar);
				ofstrm.close();
			    }
			    delete part;
			}
		    } // for (i = 0, ...
		    // letzten rest noch ausgeben
		    if (result) {
			sprintf(output, "%s_groesser_%g", outfile.c_str(),
				cluster_buckets[i-1]);
			ofstrm.open(output, ios::app);
			if (ofstrm.fail()) {
			    cerr << "Fehler beim oeffnen von " << output
				 << " Standardausgabe wird benutzt.\n";
			    gff.write(result, cout, seqname.c_str(), sourceChar);
			} else {
			    gff.write(result, ofstrm, seqname.c_str(), sourceChar);
			    ofstrm.close();
			}
		    }
		} else { // keine cluster-Bildung, "normale" Ausgabe
		    if (outfile.length()) {
			ofstrm.open(outfile.c_str(), ios::app);
			if (ofstrm.fail()) {
			    cerr << "Fehler beim oeffnen von " << outfile
				 << " Standardausgabe wird benutzt.\n";
			}
		    }
		    if (ofstrm.is_open()){
			gff.write(result, ofstrm, seqname.c_str(), sourceChar);
			ofstrm.close();
		    } else
			gff.write(result, cout, seqname.c_str(), sourceChar);
		}
		delete result;
	    } // Stueckelungsschleife
	} // loop over sequences
    } catch (DBAccessError &err) {
	cerr << "Error in the input file:\n" << err.getMessage() << endl;
	return 1;
    }
    return 0;
} // main
    
