/*****************************************************************************
 ** Dateiname:    DBProcess.hh
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Steuerung der verschieden Abfragemoeglichkeiten
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 9.12.2002  | Oliver Schoeffmann      | Erzeugung der Datei (nur mit EST)
 ** 19.2.2003  | Oliver Schoeffmann      | Hinzufuegen der Proteinverarbeitung
 ** 2.4.2003   | Oliver Schoeffmann      | Hinzufuegen der kombinierten Abfrage
 *****************************************************************************/

#ifndef __dbprocess_hh__
#define __dbprocess_hh__

#include <string>

#include <DBError.hh>
#include <DBMatch.hh>
#include <DBProperties.hh>
using namespace std;

//
// Klasse DBProcessError
//
// Fehlerklasse von DBProcess, abgeleitet von DBError
// Die Art des Fehlers wird durch "m" mitgeteilt

class DBProcessError : public DBError {

public:
    DBProcessError(const string m) : DBError(m) {}

}; // class DBProcessError


// Klasse: DBProcess
//
// Sie bietet Methoden fuer die verschiedenen Ablaufmoeglichkeiten
// von AGRIPPA

class DBProcess {
public:

    // Konstruktor (default)

    DBProcess();

    // Konstruktor
    //
    // Parameter: Ein Objekt vom Typ 'DBProperties'

    DBProcess(DBProperties properties);

    // Destruktor
    
    ~DBProcess();


    // Methode: setProperties
    //
    // Beschreibung: Weist der Klasse ein (neues) DBProperties Objekt zu
    //
    // Parameter: Ein Objekt vom Typ 'DBProperties'
    
    void setProperties(DBProperties properties);


    // Methode: process
    //
    // Beschreibung: Veranlasst eine kombinierte Ergebnisverarbeitung.
    //
    // Parameter: 'sequence' - Die zu untersuchende Sequenz
    //            'estfile'  - Eine Datei, in der schon die Suchergebnisse
    //                         einer EST-Suche enthalten sind.
    //                         Default: 0 (EST-Suche selbst durchfuehren)
    //
    // Rueckgabewert: Eine Liste der Ergebnisse
    //
    // Exceptions: keine Sequenz, Fehler bei der Datenbanksuche

    DBMatch_t *process(const char *sequence,
		       const char *estfile = 0) throw (DBProcessError);


    // Methode: processEST
    //
    // Beschreibung: Veranlasst eine EST-Verarbeitung
    //
    // Parameter: 'sequence' - Die zu untersuchende Sequenz
    //            'filename' - Eine Datei, in der schon die Suchergebnisse
    //                         einer EST-Suche enthalten sind.
    //                         Default: 0 (EST-Suche selbst durchfuehren)
    //            'single'   - true: EST-Treffer werden einzeln betrachtet
    //                         false: EST-Treffer werden gemischt
    //                         Default: false.
    //            'cluster'  - true: EST-Treffer werden in verschiedene Gruppen
    //                               (nach e-Wert) einsortiert.
    //                         false: alle EST-Treffer werden gemeinsam 
    //                                betrachtet
    //                         Default: false.
    //
    // Rueckgabewert: Eine Liste der Ergebnisse
    //
    // Exceptions: keine Sequenz, Fehler bei der Datenbanksuche

    DBMatch_t *processEST(const char *sequence, 
			  const char *filename = 0,
			  bool single = false,
			  bool cluster = false) throw (DBProcessError);


    // Methode: processPROT
    //
    // Beschreibung: Veranlasst eine Protein-Verarbeitung
    //
    // Parameter: 'sequence' - Die zu untersuchende Sequenz
    //            'filename' - Eine Datei, in der schon die Suchergebnisse
    //                         einer Protein-Suche enthalten sind.
    //                         Default: 0 (Protein-Suche selbst durchfuehren)
    //            'cluster'  - true: Protein-Treffer werden in verschiedene 
    //                               Gruppen (nach e-Wert) einsortiert.
    //                         false: alle Protein-Treffer werden gemeinsam 
    //                                betrachtet
    //                         Default: false.
    //
    // Rueckgabewert: Eine Liste der Ergebnisse
    //
    // Exceptions: keine Sequenz, Fehler bei der Datenbanksuche

    DBMatch_t *processPROT(const char *sequence, 
			   const char *filename = 0, 
			   bool cluster = false) throw (DBProcessError);

private:
    // Private - Variable
    DBProperties props;
}; // class DBProcess

#endif
