/*****************************************************************************
 ** Dateiname:    DBProt.cc
 ** Projekt:      Gewinnung extrinsischer Informationen durch Datenbankabfragen
 ** Beschreibung: Bietet die Proteinverarbeitung an
 ** Autor:        Oliver Schoeffmann
 **
 ** Copyright:    @Schoeffmann
 **
 ** Datum      | Autor                   | Beschreibung
 ** --------------------------------------------------------------------------
 ** 18.2.2003  | Oliver Schoeffmann      | Erzeugung der Datei
 ** 2.4.2003   | Oliver Schoeffmann      | Auslagern der kombinierten
 **                                      | Datenbankabfrage
 ** 12.4.2003  | Oliver Schoeffmann      | Bewertung der Introns ueber die 
 **                                      | beiden Trefferzeilen
 *****************************************************************************/

#include <iostream>
#include <cstring>
#include <cstdio>
#include <unistd.h>

#include <DBBlast.hh>
#include <DBProt.hh>
#include <DBStore.hh>

using namespace std;

///////////////////////////////////////////////////////////////////////////////
// Public - Methoden
///////////////////////////////////////////////////////////////////////////////

// Konstruktor

DBProt::DBProt(DBProperties properties) {
    setProperties(properties);
}

//---------------------------------------------------------------------------//

// Konstruktor
//
// Beschreibung: einige Variablen benoetigen Startwerte

DBProt::DBProt() {
    s_ignore = 0;
    s_has = 0;
    threshold = 50;
    ratio = 0.8;
    cutoff = 4;
}

//---------------------------------------------------------------------------//

// Destruktor
//
// Gibt den Speicher von zwei Variablen wieder frei

DBProt::~DBProt() {
    delete s_ignore;
    delete s_has;
}

//---------------------------------------------------------------------------//

// Methode: pureProcess
//
// Beschreibung: s. Headerdatei

DBMatch_t *DBProt::pureProcess(const char *sequence,
			       const char *protfile,
			       const bool cluster) throw (DBProtError) {
    if (!sequence)
	throw DBProtError("DBProt::pureProcess: keine Sequenz angegeben");

    // Sequenz fuer alle Methoden zugaenglich machen
    seq = sequence;

    DBMatch_t *ret;

    // Datenbankabfrage durchfuehren bzw. die uebergebene Datei verarbeiten
    DBBlast blast(props);
    try {
      if (protfile && access(protfile, R_OK)==0) // already have the blast results
	    ret = blast.parseFile(protfile);
      else {
	if (protfile)              // no results yet, but save them once you have them
	  ret = blast.process(sequence, protfile, PROT); 
	else                               // dont save results
	  ret = blast.process(sequence, PROT);
      }
    } catch (DBBlastError &err) {
	throw DBProtError((string)"DBProt::pureProcess: "
			  "Fehler produziert durch:\n"
			  + err.getMessage());
    }
    DBStore store;

    // Alle Datenbanktreffer bearbeiten
    for (DBMatch_t *m = ret; m; m = m->next) {
	if (!namecheck(m->name))
	    continue;
	if (cluster) {
	    clustern(m);
	}
	// Treffer bearbeiten
	DBMatch_t *tmp = cleanup(m);
	if (!tmp) {
	    delete tmp;
	    continue;
	}
	try {
	    
	    // Einsortieren der bearbeiteten Treffer
	    for (DBMatch_t *takt = tmp; takt; takt = takt->next) {
		if (!store.cutWith(takt, cluster))
		    store.add(takt, cluster, true);
	    }
	} catch (DBStoreError &err) {
	    throw DBProtError((string)"DBProt::pureProcess: "
			      "Fehler produziert durch:\n"
			      +err.getMessage());
	}
	delete tmp;
    } // for ...
    delete ret;
    try {
	ret = store.getMatches();
	// die richtigen Frames in den Treffern setzen
	setFrames(ret);
    } catch (DBStoreError &err) {
	throw DBProtError((string)"DBProt::pureProcess: "
			  "Fehler produziert durch:\n"
			  +err.getMessage());
    }
    /*
    cout << "******* alle guten Treffer" << endl;
    for (DBMatch_t *treffer = ret; treffer != NULL; treffer=treffer->next) {
      cout <<  "Treffer" <<*treffer << endl;
      }*/
    return ret;
} // pureProcess

//---------------------------------------------------------------------------//

// Methode: setProperties
//
// Beschreibung: s. Headerdatei

void DBProt::setProperties(DBProperties properties) {
    props = properties;
    readProps();
} // setProperties


///////////////////////////////////////////////////////////////////////////////
// Private - Methoden
///////////////////////////////////////////////////////////////////////////////

// Private - Methode: cleanup
//
// Beschreibung: Verarbeitet die Datenbanktreffer und bildet Komplexe aus den 
//               Teiltreffern.
// Parameter: 'm' - Der zu bearbeitende Treffer
//
// Rueckgabewert: Liste der Komplexe.
//
// Exception: Kopierfehler

DBMatch_t *DBProt::cleanup(DBMatch_t *m) throw (DBProtError) {
    if (!m || !m->parts)
	return 0;

    // Eine Kopie des Treffers wird angefertigt
    DBMatch_t *ret;
    try {
	ret = m->copy();
    } catch (DBMatchError &err) {
	throw DBProtError((string)"DBProt::cleanup: Fehler verursacht durch:\n"
			  + err.getMessage());
    }

    // Die Teiltreffer werden an den Raender bearbeitet
    smooth(ret);

    // Evtl. "eingeschlossene" Introns innerhalb der Teiltreffer werden gesucht
    includedIntrons(ret);

    // Die Teiltreffer werden "bewertet"
    rateHits(ret);
    if (!ret || !ret->parts)
	return 0;
    // Die Teiltreffer werden zu Komplexen zusammengefasst
    DBMatch_t  *retp = 0, *rakt = 0;
    while (ret->parts) {
	DBMatch_t *tmp = findComplex(ret);
	if (tmp) {
	    if (tmp->wholeScore < threshold) {
		delete tmp;
		continue;
	    }
	    if (retp) {
		rakt->next = tmp;
		rakt = rakt->next;
	    } else {
		rakt = retp = tmp;
	    }
	}
    }
    delete ret;
    return retp;
} // cleanup

//---------------------------------------------------------------------------//

// Private - Methode: clustern
//
// Beschreibung: Der Treffer wird in eine bestimmte Klasse eingeteilt 
//               (nach e-Wert). Das Element 'cluster' des Treffers wird
//               dementsprechend veraendert.
//
// Parameter: 'm' - Der zu bearbeitende Treffer.

void DBProt::clustern(DBMatch_t *m) {
    if (cluster_buckets) {
	for (int i = 0; cluster_buckets[i] != PROP_DOUBLE_END; i++) {
	    if (m->cluster <= cluster_buckets[i]) {
		m->cluster = cluster_buckets[i];
		return;
	    }
	}
	m->cluster = PROP_DOUBLE_END;
    }
} // clustern

//---------------------------------------------------------------------------//

// Private - Methode: smooth
//
// Beschreibung: Glaettet die Raender der Teiltreffer
//
// Parameter: 'm' - Treffer mit den Teiltreffern

void DBProt::smooth(DBMatch_t *m) {
    DBParts_t *akt = m->parts, *pre = 0;
    int trend = m->trend;
    while (akt) {
	string &hitline = akt->hitline;
	int length = hitline.length(), cut = 0, sbj = 0, qry = 0, minus = 0;

	// fuehrende '-' in sbj oder qry werden abgeschnitten
	// sind in den ersten 10 Zeichen mehr als 4 Mismatches, so wird auch // war 3 vorher
	// das erste Zeichen entfernt
	char last = hitline[0];
	for (int i = 0; i < 10; i++)
	    if (hitline[i] == '-')
		minus++;
	
	// "> 10" braucht das naechste 'for'
	while (length - cut > 10) {
	    if (akt->query[cut] != '-'   &&
		akt->subject[cut] != '-' &&
		minus <= 4)
		break;
	    if (last != hitline[10 + cut])
		minus += last == '-' ? -1 : 1;
	    // nur wenn in den Zeilen ein Buchstabe ist, werden die 
	    // Grenzen veraendert
	    if (akt->subject[cut] != '-')
		sbj++;
	    if (akt->query[cut] != '-')
		qry++;
	    cut++;
	    last = hitline[cut];
	}
	hitline.erase(0, cut);
	akt->query.erase(0, cut);
	akt->subject.erase(0, cut);
	akt->qry_from += qry * 3;
	akt->sbj_from += trend * sbj;
	length -= cut;

	// abschliessende '-' in sbj oder qry werden abgeschnitten
	last = hitline[length - 1];
	sbj = qry = minus = cut = 0;
	for (int i = 0; i < 10; i++)
	    if (hitline[length - i - 1] == '-')
		minus++;
	while (length - cut >= 10 && minus > 4) {
	    if (last != hitline[length - 11 - cut])
		minus += last == '-' ? -1 : 1;
	    if (akt->subject[length - cut - 1] != '-')
		sbj++;
	    if (akt->query[length -cut - 1] != '-')
		qry++;
	    cut++;
	    last = hitline[length - cut - 1];
	}
	length -= cut;
	hitline.erase(length);
	akt->query.erase(length);
	akt->subject.erase(length);
	akt->qry_to -= 3 * qry;
	akt->sbj_to -= trend * sbj;

	// Stuecke die < 10 Zeichen sind werden entfernt
	if (length < 10) {
	    if (pre) {
		pre->next = akt->next;
		akt->next = 0;
		delete akt;
		akt = pre->next;
	    } else {
		m->parts = akt->next;
		akt->next = 0;
		delete akt;
		akt = m->parts;
	    }
	} else {
	    pre = akt;
	    akt = akt->next;
	}
    }
} // smooth

//---------------------------------------------------------------------------//

// Private - Methode: includedIntrons
//
// Beschreibung: Sucht in den Teiltreffern enthaltene Introns und splittet die 
//               Teiltreffer auf.
// 
// Parameter: 'm' - Treffer mit den Teiltreffern
//
// Exception: Kopierfehler

void DBProt::includedIntrons(DBMatch_t *m) throw (DBProtError) {
    if (m->parts == 0)
	return;
    for (DBParts_t *mp = m->parts; mp; mp = mp->next) {
	long exonend = 0;
	int gap = 0;
	int length = mp->subject.length();

	// Bestimme in einem Teiltreffer lange Abschnitte von '-' in 
	// der Datenbanksequenz (subject)
	for (long i = 0; i < length; i++) {
	    if (mp->subject[i] == '-') {
		gap++;
	    } else {
		// mindestens 48 bp langes Intron (16 Codons)
		if (gap >= 16) {

		    // Hinweise auf ein Intron gefunden. Der Teiltreffer
		    // wird aufgeteilt
		    DBParts_t *tmp;
		    try {
			tmp = mp->copy();
		    } catch (DBMatchError &err) {
			throw DBProtError((string)"Fehler verursacht durch:\n"
					  + err.getMessage());
		    }
		    mp->sbj_to = mp->sbj_from + m->trend * exonend;
		    mp->qry_to = mp->qry_from + 3 * (exonend + 1) - 1;
		    mp->subject.erase(exonend + 1);
		    mp->hitline.erase(exonend + 1);
		    mp->query.erase(exonend + 1);
		    tmp->next = mp->next;
		    mp->next = tmp;
		    tmp->sbj_from = mp->sbj_to + m->trend;
		    tmp->qry_from = mp->qry_to + 3 * gap + 1;
		    tmp->subject.erase(0, exonend + gap + 1);
		    tmp->hitline.erase(0, exonend + gap + 1);
		    tmp->query.erase(0, exonend + gap + 1);
		    break;
		} else {
		    gap = 0;
		    exonend = i;
		}
	    }
	}
    }
} // includedIntrons

//---------------------------------------------------------------------------//

// Private - Methode: findComplex
//
// Beschreibung: Erstellt aus den (noch vorhandenen) Teiltreffern einen
//               Komplex. Der erste dieser Teiltreffer "beginnt" den neuen 
//               Komplex.
//
// Parameter: 'm' - Treffer mit den Teiltreffern
//
// Rueckgabewert: gebildeter Komplex.
//
// Nebeneffekt: Die verwendeten Teiltreffer werden aus 'm' entfernt.
//
// Exception: Speicherfehler

DBMatch_t *DBProt::findComplex(DBMatch_t *m) throw (DBProtError) {
    DBParts_t *akt = m->parts, *preakt = 0;
    m->parts = m->parts->next;
    akt->next = 0;

    int trend = m->trend;
    m->wholeScore -= akt->score;

    // Der erste Teiltreffer wird auf ein Start- / Stoppcodon hin untersucht
    // (je nach Orientierung) und evtl. gekuerzt (um cutoff)
    if (trend == 1) {
	if (akt->sbj_from == 1 && isStart(akt->qry_from, false))
	    akt->start = true;
	else
	    akt->qry_from += 3 * cutoff;
    } else {
	if (akt->sbj_from == m->wholeLength && isEnd(akt->qry_from, true)) {
	    akt->end = true;
	    akt->qry_from -= 3;
	}
	else
	    akt->qry_from += 3 * cutoff;
    }

    // Ist der erste Teiltreffer entfernt worden, gibt es keinen Komplex
    if (akt->qry_from >= akt->qry_to)
	return 0;

    DBMatch_t *ret = new DBMatch_t;
    if (!ret)
	throw DBProtError("DBProt::cleanup: Speicherfehler");
    
    // ret-Werte setzen
    ret->name = new char[strlen(m->name) + 1];
    if (!ret->name)
	throw DBProtError("DBProt::cleanup: Speicherfehler");
    strcpy(ret->name, m->name);
    ret->wholeLength = m->wholeLength;
    ret->trend = m->trend;
    ret->cluster = m->cluster;
    ret->parts = akt;
    ret->wholeScore = akt->score;

    // Es werden passende Teiltreffer fuer den Komplex gesucht
    DBParts_t *p = 0, *pre = 0;
    do {
	for (p = m->parts, pre = 0; p; pre = p, p = p->next) {
	    if (hasIntron(akt, p, trend)) {
		// Intron zwischen zwei Teiltreffern gefunden
		preakt = akt;
		akt->next = p;
		akt = akt->next;
		ret->wholeScore += akt->score;
		m->wholeScore -= akt->score;
		if (pre) 
		    pre->next = p->next;
		else 
		    m->parts = m->parts->next;
		akt->next = 0;
		//intron = true;
		break;
	    }
	} 
    } while (p); // ende wenn for-schleife erfolglos

    // Letzter Teiltreffer wir wie oben behandelt
    if (trend == -1) {
	if (akt->sbj_to == 1 && isStart(akt->qry_to, true))
	    akt->start = true;
	else
	    akt->qry_to -= 3 * cutoff;
    } else {
	if (akt->sbj_to == m->wholeLength && isEnd(akt->qry_to, false)) {
	    akt->end = true;
	    akt->qry_to += 3;
	}
	else
	    akt->qry_to -= 3 * cutoff;
    }
    
    if (akt->qry_from >= akt->qry_to) {
	if (preakt) {
	    preakt->next = 0;
	    ret->wholeScore -= akt->score;
	    delete akt;
	} else {
	    delete ret;
	    ret = 0;
	}
    }

    return ret;
} // findComplex

//---------------------------------------------------------------------------//

// Public - Methode: hasIntron
//
// Beschreibung: Zwischen zwei Teiltreffern wird ein Intron gesucht.
//
// Parameter: 'p'     - "erster" Teiltreffer
//            'q'     - "zweiter" Teiltreffer
//            'trend' - Vorgabe fuer die Orientierung des Intron
//
// Rueckgabewert: 'true'  - Ein Intron wurde gefunden.
//                'false' - Kein Intron wurde gefunden.

bool DBProt::hasIntron(DBParts_t *p, 
		       DBParts_t *q, 
		       int trend) {
    long psto = p->sbj_to, qsfrom = q->sbj_from;
    long pqto = p->qry_to, qqfrom = q->qry_from;
    int diff = trend * (psto - qsfrom) + 1;

    // In den folgenden Faellen kann es kein Intron geben.
    if (diff < 0                                       || // klar
	p->qry_from > qqfrom                           || // 2. Start vor 1.
	pqto > q->qry_to                               || // 1. Ende vor 2.
	3 * diff < (pqto - qqfrom + 1)                 || // qschnitt > 3*diff
	3 * diff >= (q->qry_to - qqfrom + 1)           || // schnitt groesser..
	3 * diff >= (pqto - p->qry_from + 1)           || // .. als qlaenge
	diff >= trend * (p->sbj_to - p->sbj_from) + 1  || // schnitt groesser..
	diff >= trend * (q->sbj_to - q->sbj_from) + 1) {  // .. als slaenge
	return false;
    }

    // Die "Splicesites"
    char ass1 = 'a', ass2 = 'g', dss1 = 'g', dss2 = 't';
    if (trend < 0) {
      dss1 = 'c'; // Mario: Looks wrong but does the right thing.
      ass2 = 'c'; // The names are just swapped: dss is now the reverse complement of the ass
    }

    // Die Raender der Teiltreffer, an denen ein Intron gesucht wird, werden
    // nochmals bearbeitet
    backcut(p, diff);
    pqto = p->qry_to;
    frontcut(q, diff);
    qqfrom = q->qry_from;


    if (diff < 0)
	return false;

    // Die eigentliche Intronsuche
    int qdiff = diff * 3;
    int count = 0, maxplus = -1, imark = 0;
    if (diff == 0) {
	for (int i = -3; i <= 3; i++) {
	    if (seq[pqto - i] == dss1 && 
		seq[pqto - i + 1] == dss2 &&
		seq[qqfrom - i - 2] == ass2 && 
		seq[qqfrom - i - 3] == ass1) {
		maxplus = 0;
		imark = i;
		break;
	    }	    
	}
    } else {
	for (int i = -3; i <= qdiff + 3; i++) {
	    if (seq[pqto - i] == dss1 && 
		seq[pqto - i + 1] == dss2 &&
		seq[qqfrom + qdiff - i - 2] == ass2 && 
		seq[qqfrom + qdiff - i - 3] == ass1) {
		// Ein moegliches Intron ist gefunden worden
		int plus = 0;

		// Bewertung des Introns
		int fcomp = (qdiff - i) / 3;
		int bcomp = i / 3;
		for (int k = 1; k <= fcomp; k++)
		    if (p->hitline[p->hitline.length() - diff - 1 + k] == '+')
			plus++;
		for (int k = 1; k <= bcomp; k++)
		    if (p->hitline[diff - k] == '+')
			plus++;
		if (plus == maxplus)
		    count++;
		if (plus > maxplus) {
		    count = 1;
		    maxplus = plus;
		    imark = i;
		}
	    }
	}
    }
    if (maxplus == -1)
	return false;

    // Anpassen der Teiltreffer
    p->qry_to -= imark;
    q->qry_from += qdiff - imark;
 
    return true;
} //hasIntron

//---------------------------------------------------------------------------//

// Private - Methode: rateHits
//
// Beschreibung: Nur Teiltreffer von bestimmter Guete werden genommen.
//
// Parameter: 'm' - Treffer mit den zu untersuchenden Teiltreffern

void DBProt::rateHits(DBMatch_t *m) {
    DBParts_t *mp = m->parts, *akt = 0, *tmp = 0;
    m->parts = 0;
    while (mp) {
	unsigned length = mp->hitline.length();
	tmp = mp;
	mp = mp->next;
	tmp->next = 0;
	int minus = 0;

	// Die Anzahl der Positionen, an denen beiden Sequenzabschnitte
	// nicht uebereinstimmen, wird bestimmt
	for (unsigned i = 0; i < length; i++)
	    if (tmp->hitline[i] == '-')
		minus++;

	// Ist ein Teiltreffer zu "schlecht", wird er geloescht.
	if ((1 - ratio) < (double) minus / length) {
	    delete tmp;
	} else {
	    if (akt) {
		akt->next = tmp;
		akt = akt->next;
	    } else {
		m->parts = tmp;
		akt = tmp;
	    }
	}
    }
} // rateHits

//---------------------------------------------------------------------------//

// Private - Methoden: frontcut / backcut
//
// Beschreibung: Untersucht die Datenbanksequenz auf '-' und entfernt diese 
//               Positionen. Es treten sonst Verfaelschungen bei der 
//               Intronsuche auf.
//
// Parameter: 'q'/'p' - Der zu untersuchende Teiltreffer
//            'diff'  - Die Laenge des zu untersuchenden Abschnittes
//
// Nebeneffekt: 'diff' wird veraendert, wenn solche Positionen gefunden werden.

void DBProt::frontcut(DBParts_t *q, 
		      int &diff) {
    for (int i = 0, n = 0; n <= diff; n++, i++) {
	if (q->subject[i] == '-') {
	    if (q->query[i] != '-')
		q->qry_from += 3;
	    q->query.erase(i, 1);
	    q->subject.erase(i, 1);
	    q->hitline.erase(i, 1);
	    i--;
	    //diff--;
	} else if (q->query[i] == '-')
	    q->qry_from -= 3;

    }
} // frontcut

//---------------------------------------------------------------------------//

void DBProt::backcut(DBParts_t *p,
		     int &diff) {
    int length = p->subject.length();
    for (int i = 0, n = 0; n <= diff; n++, i++) {
	if (p->subject[length - i - 1] == '-') {
	    if (p->query[length - i - 1] != '-') {
		p->qry_to -= 3;
	    }
	    p->subject.erase(length - i - 1, 1);
	    p->query.erase(length - i - 1, 1);
	    p->hitline.erase(length - i - 1, 1);
	    length--;
	    i--;
	    //diff--;
	} else if (p->query[length - i - 1] == '-')
	    p->qry_to += 3;
    }
} // backcut

//---------------------------------------------------------------------------//

// Private - Methode: isStart
//
// Beschreibung: Ueberprueft die angegebene Position auf ein Startcodon
//               Die angegebene Position ist dabei das erste Nukleotid 
//               des zu untersuchenden Codons (in Leserichtung des Codons).
//
// Parameter: 'start'   - Startposition des zu untersuchenden Codons
//            'reverse' - Angabe der Leserichtung: 
//                        true  = Vorw"artsstrang,
//                        false = Gegenstrang
//
// Rueckgabewert: true, wenn ein Startcodon in der angegebenen Lererichtung
//                gefunden wurde sonst false.
//
// Bemerkung: Positionsangabe ist NICHT C-like. Daher ist stets eine Position 
//            abzuziehen

bool DBProt::isStart(long start,
		     bool reverse) {
    if (reverse) {
	if (start > 2)
	    return (seq[start - 1] == 't' && seq[start - 2] == 'a'
		    && seq[start - 3] == 'c');
	else
	    return false;
    }
    if (start < (long)strlen(seq) - 1)
	return (seq[start - 1] == 'a' && seq[start] == 't'
		&& seq[start + 1] == 'g');
    else 
	return false;
} // isStart

//---------------------------------------------------------------------------//

// Private - Methode: isEnd
//
// Beschreibung: Ueberprueft die angegebene Position auf ein Stoppcodon
//               Die angegebene Position ist dabei vor dem ersten Nukleotid 
//               des zu untersuchenden Codons (in Leserichtung des Codons).
//
// Parameter: 'end'     - Position vor dem zu untersuchenden Codon
//            'reverse' - Angabe der Leserichtung: 
//                        true  = Vorw"artsstrang,
//                        false = Gegenstrang
//
// Rueckgabewert: true, wenn ein Stoppcodon in der angegebenen Lererichtung
//                gefunden wurde sonst false.
//
// Bemerkung: Positionsangabe ist NICHT C-like. Daher ist stets eine Position 
//            abzuziehen

bool DBProt::isEnd(long end, 
		   bool reverse) {
    // possible: crossing seq-bounds
    if (reverse) {
	if (end > 3)
	    return (seq[end - 2] == 'a' &&
		    ((seq[end - 3] == 't' &&
		      (seq[end - 4] == 't' || seq[end - 4] == 'c')) ||
		     (seq[end - 3] == 'c' && seq[end - 4] =='t')));
	else
	    return false;
    }
    if (end < (long)strlen(seq) - 2)
	return (seq[end] == 't' &&
		((seq[end + 1] == 'a' &&
		  (seq[end + 2] == 'g' || seq[end + 2] == 'a')) ||
		 (seq[end + 1] == 'g' && seq[end + 2] == 'a')));
    else
	return false;
} // isEnd

//---------------------------------------------------------------------------//

// Private - Methode: setFrames
//
// Beschreibung: Setzt die Leserahmen der Teiltreffer.
//               Der Leserahmen haengt von der Laenge und des Leserahmens
//               des vorherigen (in Leserichtung) Teiltreffers ab. Der
//               erste Teiltreffer besitzt den Leserahmen 0.
//
// Parameter: 'm' - Liste der Treffer, deren Teiltreffer bearbeitet werden

void DBProt::setFrames(DBMatch_t *m) {
    if (!m || !m->parts)
	return;
    for (DBMatch_t *t = m; t; t = t->next) {

	// Unterschiedliche Vorgehensweisen bie unterschiedlicher Orientierung
	// der Treffer.
	if (t->trend < 0) {

	    // Treffer auf dem Gegenstrang.
	    // Die Teiltreffer muessen "von hinten nach vorne" bewertet werden.
	    // Dazu werden zuerst alle Teiltreffer ausser dem "ersten" in der
	    // Liste (von der Leserichtung her der "letzte")bewertet. 
	    // Daraus ergibt sich der Leserahmen des "ersten" Teiltreffers und 
	    // die anderen Leserahmen lassen sich daraus bestimmen.
	    int lastframe = 0;
	    for (DBParts_t *p = t->parts->next; p; p = p->next) {
		lastframe += (3 - (p->qry_to - p->qry_from + 1) % 3) % 3;
	    }
	    lastframe %= 3;
	    t->parts->frame = lastframe;
	    for (DBParts_t *p = t->parts->next; p; p = p->next) {
		p->frame =  (p->qry_to - p->qry_from + 1 + lastframe) % 3;
		lastframe = p->frame;
	    }
	} else {

	    // Treffer auf dem Vorwaertsstrang.
	    // Daraus ergibt sich eine einfache Vorgehensweise.
	    int nextframe = 0;
	    for (DBParts_t *p = t->parts; p; p = p->next) {
		p->frame = nextframe;
		nextframe = (3 -
			     (p->qry_to - p->qry_from + 1 - p->frame)% 3) % 3;
	    }
  	}
    }
}

//---------------------------------------------------------------------------//

// Private - Methode: readProps
//
// Beschreibung: Fuer die Klasse relevante Optionen werden der 
//               Konfigurationsdatei (durch die Klasse DBProperties) entnommen.
//               Liegen keine Angaben ueber die einzelnen Variablen vor, 
//               so werden Default-Werte gesetzt.

void DBProt::readProps(void) {
    try {
        s_ignore = props.getStringArray("prot_ignore_string");
        s_has = props.getStringArray("prot_has_string");
    } catch (DBPropertiesError &err) {
        cerr << "DBProt::readProps: error fetching has / ignore strings\n"
             << "proceed with none of them.\n";
        s_has = s_ignore = 0;
    }
    try {
	cluster_buckets = props.getDoubleArray("cluster_buckets");
    } catch(DBPropertiesError &err) {
	cerr << "Fehler beim Lesen von "
	     << "\"prot_ignore/has_string\""
	     << "\nWeitere Bearbeitung ohne \"cluster_buckets\"\n";
	cluster_buckets = 0;
    }

    // andere defaultwerte
    try {
        threshold = props.getInt("prot_threshold");
    } catch (DBPropertiesError &err) {
        threshold = 50;
    }
    try {
        cutoff = props.getInt("prot_cutoff");
    } catch (DBPropertiesError &err) {
        cutoff = 4;
    }
    try {
        ratio = props.getDouble("prot_ratio");
    } catch (DBPropertiesError &err) {
        ratio = 0.8;
    }
    if (ratio < 0.0 || ratio > 1.0) {
	cerr << "Unguelitige Angabe fuer \"prot_ratio\". Setze default 0.8\n";
        ratio = 0.8;
    }
} // readProps

//---------------------------------------------------------------------------//

// Private - Methode: namecheck
//
// Beschreibung: Ueberprueft den Bezeichner auf bestimmt Worte.
//              
// Parameter: 'name' - Der zu untersuchende Bezeichner
//
// Rueckgabewert: false - Wenn in dem Bezeichner eines der Worte aus 's_ignore'
//                        oder nicht alle Worte aus 's_has' gefunden wurden.
//                true  - sonst

bool DBProt::namecheck(const char *name){
    if (s_ignore)
        for (int i = 0; s_ignore[i] != PROP_STRING_END; i++)
            if (strstr(name, s_ignore[i].c_str()) != NULL)
                return false;
    if (s_has)
        for (int i = 0; s_has[i] != PROP_STRING_END; i++)
            if (strstr(name, s_has[i].c_str()) == NULL)
                return false;
    return true;
} // namecheck

