/*
 * 	train_labelling.h
 *
 *  Created on: June 8th, 2011
 *      Author: torsten
 */

#include "train_labelling.h"
#include "train_types.h"
#include "train_eval.h"
#include <graph_types.h>
#include <fea/fea_histogram.h>

#include <dai/factor.h>
#include <dai/bp.h>
#include <dai/gibbs.h>
#include <dai/factorgraph.h>
#include <dai/graph.h>

#include <vector>
#include <map>
#include <math.h>
#include <assert.h>
#include <algorithm>
#include <fstream>
#include <stdlib.h>
#include <string>
#include <time.h>

#ifdef DEBUG_TRAIN_LABELLING
extern
void	train_labelling_log	(char*);
static
char	log_txt	[4 * 4096];
#define	LOG_TRAIN_LABELLING(s) sprintf(log_txt,s); \
							if(Classifier::Verbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_2(s,t) sprintf(log_txt,s,t); \
							if(Classifier::Verbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_3(s,t,u) sprintf(log_txt,s,t,u); \
							if(Classifier::Verbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_4(s,t,u,v) sprintf(log_txt,s,t,u,v); \
							if(Classifier::Verbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_5(s,t,u,v,w) sprintf(log_txt,s,t,u,v,w); \
							if(Classifier::Verbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_6(s,t,u,v,w,x) sprintf(log_txt,s,t,u,v,w,x); \
							if(Classifier::Verbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_7(s,t,u,v,w,x,y) sprintf(log_txt,s,t,u,v,w,x,y); \
							if(Classifier::MoreVerbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_9(p,q,r,s,t,u,v,w,x) sprintf(log_txt,p,q,r,s,t,u,v,w,x); \
							if(Classifier::MoreVerbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_10(p,q,r,s,t,u,v,w,x,y) sprintf(log_txt,p,q,r,s,t,u,v,w,x,y); \
							if(Classifier::MoreVerbose()) train_labelling_log(log_txt);
#define	LOG_TRAIN_LABELLING_11(p,q,r,s,t,u,v,w,x,y,z) sprintf(log_txt,p,q,r,s,t,u,v,w,x,y,z); \
							if(Classifier::MoreVerbose()) train_labelling_log(log_txt);
#else

#define	LOG_TRAIN_LABELLING(s)
#define	LOG_TRAIN_LABELLING_2(s,t)
#define	LOG_TRAIN_LABELLING_3(s,t,u)
#define	LOG_TRAIN_LABELLING_4(s,t,u,v)
#define	LOG_TRAIN_LABELLING_5(s,t,u,v,w)
#define	LOG_TRAIN_LABELLING_6(s,t,u,v,w,x)
#define	LOG_TRAIN_LABELLING_7(s,t,u,v,w,x,y)
#define	LOG_TRAIN_LABELLING_9(p,q,r,s,t,u,v,w,x)
#define	LOG_TRAIN_LABELLING_10(p,q,r,s,t,u,v,w,x,y)
#define	LOG_TRAIN_LABELLING_11(p,q,r,s,t,u,v,w,x,y,z)

#endif // DEBUG_TRAIN_LABELLING

namespace Training {

bool
Classifier::m_Hists_Initialized
							=	false,
Classifier::m_Verbose		=	false,
Classifier::m_MoreVerbose	=	false,
Classifier::m_Run_InLogDomain
							=	true;

std::vector<dai::Factor>
Classifier::m_Factors;

dai::FactorGraph*
Classifier::m_Fg			=	NULL;

bool
Classifier::m_Fg_Needs_Update
							=	true;

std::vector<size_t>
Classifier::m_MAP_Nodes;

Features::Histogram
Classifier::if_pssm,
Classifier::noif_pssm,
Classifier::noif_rasa,
Classifier::if_rasa,
Classifier::noif_conserv,
Classifier::if_conserv,
Classifier::noif_epros,
Classifier::if_epros,
Classifier::noif_rg,
Classifier::if_rg,
Classifier::noif_fe,
Classifier::if_fe;



size_t
Classifier::m_Nr_Nodes		=	0,
Classifier::m_Nr_Edges		=	0,
Classifier::m_LearnStep		=	0,
Classifier::m_Overall_LearnStep
							=	0;
double
Classifier::m_Initial_g_ii	=	0,
Classifier::m_Initial_g_ij	=	0,
Classifier::m_Curr_g_ii		=	0,
Classifier::m_Curr_g_ij		=	0;

size_t
Classifier::m_Structural_Diff_BigL
							=	0,
Classifier::m_Prev_Structural_Diff_BigL
							=	0,
Classifier::m_TP_ref		=	0,
Classifier::m_TN_ref		=	0,
Classifier::m_FN_curr		=	0,
Classifier::m_FP_curr		=	0;

double
Classifier::m_FP_Scale		=	1.,
Classifier::m_FN_Scale		=	1.;

std::vector<double>
Classifier::m_Initial_V,
Classifier::m_Curr_V,
Classifier::m_Diff;

size_t
Classifier::m_Opt_Imm_Rslts	=	0;

bool
Classifier::m_Opt_Output_Per_Object
							=	false,
Classifier::m_Opt_Functional_ROCdiff
							=	false,
Classifier::m_Opt_Functional_ROCcurve
							=	true,
Classifier::m_Opt_Functional_ROCset_1
							=	false,
Classifier::m_Opt_Functional_ROCset_2
							=	false;

std::vector<double>
Classifier::m_W_g,
Classifier::m_Prv_W_g,
Classifier::m_Curr_W_g,
Classifier::m_Initial_g;

double
Classifier::m_INITAL_EDGE_WEIGHT_BETA_g_ij
							=	0.5;

double
Classifier::m_INITAL_EDGE_WEIGHT_ALPHA_g_ii
							=	0.5;

bool
Classifier::m_Opt_EdgeModel_UserSupplied
							=	false,
Classifier::m_Opt_No_EdgeModel_Learning
							=	false;

const double
LEARN_RATE_G				=	0.00001,
LEARN_RATE_Q				=	0.0001,
PSEUDO_COUNT				=	0.001;

size_t
Classifier::m_Consec_NegDiffs
							=	0,
Classifier::m_Consec_ErrClassifierRuns
							=	0;
double
Classifier::m_Prev_Diff_Map	=	0.;

std::vector<double>
Classifier::m_W_q,
Classifier::m_Prv_W_q,
Classifier::m_Curr_W_q;

std::vector<size_t>
Classifier::m_Ref_Labelling;

std::string
Classifier::m_Output_Obj,
Classifier::m_Output_Dir;

size_t
Classifier::m_Opt_MaxNrLearnSteps_PerSample
							=	30,
Classifier::m_Opt_Chunk_Size=	80,
Classifier::m_Opt_LearnRound_PerInput
							=	1;

std::vector<std::vector<double>	>
Classifier::m_W_g_chunk;

std::vector<std::vector<double>	>
Classifier::m_W_q_chunk;

const
std::vector<GraphGen::Node<Types::Node>*>*
Classifier::m_Calpha_backbone
							=	NULL;

size_t
Classifier::m_Min_node_id	=	0,
Classifier::m_Max_node_id	=	0;

double
Classifier::m_SumFeat_Slot_Width
							=	0.;

bool
Classifier::m_Opt_Use_RASAext
							=	true,
Classifier::m_Opt_Use_PssmExt
							=	true,
Classifier::m_Opt_Rg		=	true,
Classifier::m_Opt_Fe		=	true,
Classifier::m_Opt_Rg_CfgData=	true,
Classifier::m_Opt_Use_ConservExt
							=	true,
Classifier::m_Opt_Use_Epros	=	true,
Classifier::m_Opt_InferWithGibbs
							=	false,
Classifier::m_Opt_Using_Exp_EnergyTerm
							=	false,
Classifier::m_Opt_BinFeatures
							=	false,
Classifier::m_SetWeightZero_iff_Negative
							=	false;

struct
FindNode	:   std::unary_function<const dai::Factor&,bool> {

		FindNode	(u_int32_t id):m_Id(id)
        {  }

        bool
        operator()  (const dai::Factor& x) {
        	bool ret(false);
        	ret = (x.vars().elements().size() == 1);
        	if(ret)
        		ret = (x.vars().front().label() == m_Id);
        	return ret;
        }

        u_int32_t m_Id;
};

void
Classifier::SetEdgeModel(double alpha, double beta) {
	m_INITAL_EDGE_WEIGHT_BETA_g_ij = beta;
	m_INITAL_EDGE_WEIGHT_ALPHA_g_ii = alpha;
	m_Opt_EdgeModel_UserSupplied = true;
}

void
Classifier::SetOpt_FP_Scale_inOLM(double scale_fp) {
	m_FP_Scale = scale_fp;
}

void
Classifier::SetOpt_FN_Scale_inOLM(double scale_fn) {
	m_FN_Scale = scale_fn;
}

size_t
Classifier::index(size_t bin, Types::Node::StateBinFeature feature) {
	size_t ext(0), base(0);
	static size_t offset_if(if_pssm.Size() + if_rasa.Size()	+ if_conserv.Size() + if_epros.Size() + if_rg.Size() + if_fe.Size());
	static size_t ext_pssm_noif(noif_rasa.Size());
	static size_t ext_conserv_noif(ext_pssm_noif + noif_pssm.Size());
	static size_t ext_epros_noif(ext_conserv_noif + noif_conserv.Size());
	static size_t ext_rg_noif(ext_epros_noif + noif_epros.Size());
	static size_t ext_fe_noif(ext_rg_noif + noif_rg.Size());

	static size_t ext_pssm_if(if_rasa.Size());
	static size_t ext_conserv_if(ext_pssm_if + if_pssm.Size());
	static size_t ext_epros_if(ext_conserv_if + if_conserv.Size());
	static size_t ext_rg_if(ext_epros_if + if_epros.Size());
	static size_t ext_fe_if(ext_rg_if + if_rg.Size());

	if(!ConfigOpt_BinFeatures())
		bin = 0;

	switch(feature) {
		case Types::Node::StateBinFeature_Feature_RASAnoif:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_NoIf;
				base = 0;
			} else
				base = offset_if;
			break;
		case Types::Node::StateBinFeature_Feature_PSSMnoif:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_NoIf;
				base = WeightOffset_NoIf * 2;
			} else {
				ext = ext_pssm_noif;
				base = offset_if;
			}
			break;
		case Types::Node::StateBinFeature_Feature_CON_noif:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_NoIf;
				base = WeightOffset_NoIf * 4;
			} else {
				ext = ext_conserv_noif;
				base = offset_if;
			}
			break;
		case Types::Node::StateBinFeature_Feature_Epros_noif:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_NoIf;
				base = WeightOffset_NoIf * 6;
			} else {
				ext = ext_epros_noif;
				base = offset_if;
			}
			break;
		case Types::Node::StateBinFeature_Feature_Rg_noif:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_NoIf;
				base = WeightOffset_NoIf * 8;
			} else {
				ext = ext_rg_noif;
				base = offset_if;
			}
			break;
		case Types::Node::StateBinFeature_Feature_Fe_noif:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_NoIf;
				base = WeightOffset_NoIf * 10;
			} else {
				ext = ext_fe_noif;
				base = offset_if;
			}
			break;
/* -- */
		case Types::Node::StateBinFeature_Feature_RASA_if:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_If;
				base = 0;
			} else
				ext = base = 0;
			break;
		case Types::Node::StateBinFeature_Feature_PSSM_if:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_If;
				base = WeightOffset_NoIf * 2;
			} else {
				base = 0;
				ext = ext_pssm_if;
			}
			break;
		case Types::Node::StateBinFeature_Feature_CON_if:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_If;
				base = WeightOffset_NoIf * 4;
			} else {
				base = 0;
				ext = ext_conserv_if;
			}
			break;
		case Types::Node::StateBinFeature_Feature_Epros_if:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_If;
				base = WeightOffset_NoIf * 6;
			} else {
				base = 0;
				ext = ext_epros_if;
			}
			break;
		case Types::Node::StateBinFeature_Feature_Rg_if:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_If;
				base = WeightOffset_NoIf * 8;
			} else {
				base = 0;
				ext = ext_rg_if;
			}
			break;
		case Types::Node::StateBinFeature_Feature_Fe_if:
			if(!ConfigOpt_BinFeatures()) {
				ext = Classifier::WeightOffset_If;
				base = WeightOffset_NoIf * 10;
			} else {
				base = 0;
				ext = ext_fe_if;
			}
			break;
		default:
			break;
	}
	return (ext + base + bin);
}

void
Classifier::Init(const std::string& indir, const bool init_for_learning) {
	size_t size_nodes_vecs(0);
	std::vector<double> w;

	srand (time(NULL));
	Eval::FullInit();

	assert(m_FP_Scale == 1. || m_FN_Scale == 1.);

	assert(noif_pssm.BuildUp(indir + "/data/2012-02-21.a/noif.pssm-imp"));
	assert(if_pssm.BuildUp(indir + "/data/2012-02-21.a/if.pssm-imp"));
	assert(noif_rasa.BuildUp(indir + "/data/2012-02-21.a/noif.rasa-imp"));
	assert(if_rasa.BuildUp(indir + "/data/2012-02-21.a/if.rasa-imp"));
	assert(noif_conserv.BuildUp(indir + "/data/2012-02-21.a/noif.conserv-imp"));
	assert(if_conserv.BuildUp(indir + "/data/2012-02-21.a/if.conserv-imp"));
	assert(noif_epros.BuildUp(indir + "/data/2013-05/2013-05-08.noif.epros"));
	assert(if_epros.BuildUp(indir + "/data/2013-05/2013-05-08.if.epros"));
	assert(noif_rg.BuildUp(indir + (ConfigOpt_Rg_CfgData() ? "/data/2013-04/2013-04-16.data-noif.rg" : "2013-04-16.keyu-noif.rg")));
	assert(if_rg.BuildUp(indir + (ConfigOpt_Rg_CfgData() ? "/data/2013-04/2013-04-16.data-if.rg" : "2013-04-16.keyu-if.rg")));
	assert(noif_fe.BuildUp(indir + "/data/2013-05/2013-05-08.noif.fe"));
	assert(if_fe.BuildUp(indir + "/data/2013-05/2013-05-08.if.fe"));

	m_Hists_Initialized = true;

	if(!m_Opt_EdgeModel_UserSupplied)
		SetEdgeModel();

	if(init_for_learning) {
		m_W_g.push_back(m_INITAL_EDGE_WEIGHT_BETA_g_ij);
		m_W_g.push_back(m_INITAL_EDGE_WEIGHT_ALPHA_g_ii);
	}
	m_Curr_W_g = m_W_g;
	m_Initial_g.push_back(1.);
	m_Initial_g.push_back(1.);

	if(!ConfigOpt_BinFeatures())
		size_nodes_vecs = Types::Node::StateBinFeature_FeatureNumber;
	else
		size_nodes_vecs = noif_pssm.Size() + if_pssm.Size() + noif_rasa.Size() + if_rasa.Size()
							+ noif_conserv.Size() + if_conserv.Size() + noif_epros.Size() + if_epros.Size()
							+ noif_rg.Size() + if_rg.Size() + noif_fe.Size() + if_fe.Size();

	if(init_for_learning) {
		m_W_q.resize(size_nodes_vecs);
		fill(m_W_q.begin(),m_W_q.end(),1.);
	}
	m_Initial_V.resize(m_W_q.size());
	m_Curr_V.resize(m_W_q.size());
	m_Diff.resize(m_W_q.size());
	m_Curr_W_q.resize(m_W_q.size());
}

void
Classifier::CleanUp() {
	m_Factors.clear();
	m_MAP_Nodes.clear();
	m_Ref_Labelling.clear();
	if(m_Fg != NULL) {
		delete m_Fg;
		m_Fg = NULL;
	}
}

void
Classifier::Purge() {
	CleanUp();
	m_W_q.clear();
	m_W_g.clear();
	m_Initial_g.clear();
	m_Initial_V.clear();
	m_Curr_V.clear();
	m_Diff.clear();
	m_Hists_Initialized = false;
}

void
Classifier::Set_Output_Object(const std::string& o) {
	m_Output_Obj = o;
}

void
Classifier::Set_Output_Dir(const std::string& d) {
	m_Output_Dir = d;
}

void
Classifier::UpdateFactors(const std::vector<GraphGen::Node<Types::Node> >& nodes) {
	double q_l0(0.), q_l1(0.);
	double g_ii(0.), g_ij(0.);
	/* update nodes factors	*/
	for(size_t k(0); k < m_Nr_Nodes; k++) {
		std::vector<GraphGen::Node<Types::Node> >::const_iterator n = find_if(nodes.begin(),nodes.end(),Types::FindNodeId(m_Factors[k].vars().front().label()));
		assert(n != nodes.end());
		MakeNodesQ(q_l0,q_l1,*n);
		m_Factors[k].set(0,q_l0);
		m_Factors[k].set(1,q_l1);
	}

	if(!m_Opt_No_EdgeModel_Learning) {
		/* update edges factors	*/
		for(size_t k(m_Nr_Nodes); k < m_Nr_Edges + m_Nr_Nodes; k++) {
			MakeEdgeG(g_ii,g_ij);
			m_Factors[k].set(0,g_ii);
			m_Factors[k].set(1,g_ij);
			m_Factors[k].set(2,g_ij);
			m_Factors[k].set(3,g_ii);
		}
	}
	m_Fg_Needs_Update = true;
}

void
Classifier::ComputeMAPmanually(const std::vector<GraphGen::Node<Types::Node> >& nodes) {
	if(MoreVerbose()) {
		if(m_Fg_Needs_Update) {
			if(m_Fg != NULL)
				delete m_Fg;
			m_Fg = new dai::FactorGraph(m_Factors);
			m_Fg_Needs_Update = false;
		}
		LOG_TRAIN_LABELLING_4("nMAP*%lu.%lu> %.5f",m_Overall_LearnStep,m_LearnStep,Eval::Score(m_MAP_Nodes))
		LOG_TRAIN_LABELLING_4("nrMAP%lu.%lu> %.5f",m_Overall_LearnStep,m_LearnStep,Eval::Score(m_Ref_Labelling))
	}
}

void
Classifier::CommonClassifiers_Setup(const std::vector<GraphGen::Node<Types::Node> >& nodes) {
    if(m_Fg_Needs_Update) {
		if(m_Fg != NULL)
			delete m_Fg;
		m_Fg = new dai::FactorGraph(m_Factors);
		m_Fg_Needs_Update = false;
    }
    Eval::Make_Reference_Labelling(nodes);
    m_MAP_Nodes.clear();
}

bool
Classifier::RunMPM(const std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges) {
    dai::PropertySet opts;
    if(m_Verbose)
    	opts.set("verbose",(size_t)1);
    else {
    	if(m_MoreVerbose)
    		opts.set("verbose",(size_t)3);
    }
    opts.set("maxiter",(size_t)6000);
    opts.set("burnin",(size_t)100);
    CommonClassifiers_Setup(nodes);

    LOG_TRAIN_LABELLING("Gibbs.")
    dai::Gibbs sample(*m_Fg,opts);
    sample.init();
    sample.run();
    m_MAP_Nodes = sample.findMaximum();
    return true;
}

bool
Classifier::RunClassifiers(const std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges, double& map_score, bool force_sampling) {
	bool ret(true);
    size_t maxiter(6000);
    double tol(1e-9);
    dai::PropertySet opts;

    opts.set("maxiter",maxiter);
    opts.set("tol",tol);
    if(m_Verbose)
    	opts.set("verbose",(size_t)1);
    else {
    	if(m_MoreVerbose)
    		opts.set("verbose",(size_t)3);
    }
    opts.set("logdomain",Run_InLogDomain());
    opts.set("damping",std::string("0.1"));
    CommonClassifiers_Setup(nodes);

    if(!force_sampling) {
		try {
			LOG_TRAIN_LABELLING("BP.")
			dai::BP bp(*m_Fg, opts("updates",std::string("SEQRND"))("inference",std::string("MAXPROD"))("logdomain",Run_InLogDomain()));
			bp.init();
			LOG_TRAIN_LABELLING_2("Props> %s",bp.printProperties().c_str())
			bp.run();
			m_MAP_Nodes = bp.findMaximum();
		} catch(dai::Exception& e) {
			LOG_TRAIN_LABELLING_5("\033[31mexcp>\033[0m%s:%s:%s:%s",e.getFilename().c_str(),e.getFunction().c_str(),e.getLine().c_str(),e.getDetailedMsg().c_str())
			ret = false;
		}
    } else {
    	try {
    		ret = RunMPM(nodes,edges);
		} catch(dai::Exception& e) {
			LOG_TRAIN_LABELLING_5("\033[31mexcp>\033[0m%s:%s:%s:%s",e.getFilename().c_str(),e.getFunction().c_str(),e.getLine().c_str(),e.getDetailedMsg().c_str())
			ret = false;
		}
    }
    if(ret) {
    	Eval::Log_ClassifierResults();
    	map_score = Eval::Score(m_MAP_Nodes);
    } else {
    	LOG_TRAIN_LABELLING("\033[41mRunClassifiers failed.\033[0m")
    }
    return ret;
}

void
Classifier::SaveBestParams(double diffmap) {
	LOG_TRAIN_LABELLING_3("\033[32mnew optimum (%lu,%lu)\033[0m",m_Overall_LearnStep,m_LearnStep)
	m_Prev_Diff_Map = diffmap;
	m_Prev_Structural_Diff_BigL = m_Structural_Diff_BigL;
	m_Prv_W_q = m_W_q;
	m_Prv_W_g = m_W_g;
	Eval::SaveBestParams(m_LearnStep);
}

void
Classifier::SetOpt_LearnFunctional_ROCset_1(double fpr, double tpr) {
	m_Opt_Functional_ROCcurve = false;
	m_Opt_Functional_ROCdiff = false;
	m_Opt_Functional_ROCset_1 = true;
	m_Opt_Functional_ROCset_2 = false;
	Eval::SetOpt_Functional_ROCset(fpr,tpr);
}

void
Classifier::SetOpt_LearnFunctional_ROCset_2(double fpr, double tpr) {
	m_Opt_Functional_ROCcurve = false;
	m_Opt_Functional_ROCdiff = false;
	m_Opt_Functional_ROCset_1 = false;
	m_Opt_Functional_ROCset_2 = true;
	Eval::SetOpt_Functional_ROCset(fpr,tpr);
}

/*	LearnStep
 * 	\param		- 	nodes
 *  \param		-	edges
 * 	\return		-	TRUE -> continue learning,
 * 					FALSE stop it
 */
bool
Classifier::LearnStep(const std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges) {
    static dai::Real pre_diff_map(0.);
    dai::Real score_refmap(0.), score_currmap(0.), diff_map(0.);
	bool ret(false), ver(Verbose());

	SetVerbose(true);
	ret = RunClassifiers(nodes,edges,score_currmap,m_Consec_NegDiffs >= CONSEC_NegDiffs_Forcing_Sampling || ConfigOpt_InferWithGibbs());
    SetVerbose(ver);

    if(ret) {
    	m_Consec_ErrClassifierRuns = 0;
		if(!m_MAP_Nodes.empty()) {

			score_refmap = Eval::Score(m_Ref_Labelling);
			diff_map = score_currmap - score_refmap;
			LOG_TRAIN_LABELLING_2("dMAPpre>%.3f",m_Prev_Diff_Map)
			LOG_TRAIN_LABELLING_2("dMAPcur>%.3f",diff_map)
			Eval::EvalMAPLabelling(nodes);
			Eval::StatsPerObject();
			ret = Compute_Differences(nodes,edges);

			if((m_Opt_Functional_ROCcurve && Eval::ComputeFunctional_ROCcurve()) ||
					(m_Opt_Functional_ROCdiff &&  Eval::ComputeFunctional_ROCdiff()) ||
					(m_Opt_Functional_ROCset_1 &&  Eval::ComputeFunctional_ROCset_1()) ||
					(m_Opt_Functional_ROCset_2 &&  Eval::ComputeFunctional_ROCset_2())) {
				SaveBestParams(diff_map);
				Eval::Save_Runs_BestWeightVectors(m_W_q,m_W_g);
			}

			if(ret && diff_map > 0.) {
			/* executes learning step by requirement
			 */
				LearnStepComputations(nodes,edges,score_currmap,score_refmap);
				ComputeMAPmanually(nodes);
				/* continue learning
				 * */
				ret = false;
				m_Consec_NegDiffs = 0;
			} else {
				if(!ret) {
				/* est(MAP) == train(MAP) => stop learning
				 * */
					LOG_TRAIN_LABELLING("\033[41mexact learned.\033[0m")
					SaveBestParams(diff_map);
					ret = true;
				} else {
				/* structural differences exists
				 * but difference in weights is <= 0
				 * => continue learning or stop depending on number
				 * 	  of consecutive negative diffs
				 */
					ret = (++m_Consec_NegDiffs > LEARN_MAX_CONSECUTE_EQ_DIFFS);

					if(m_Consec_NegDiffs > DO_RANDOM_STEP)
						DoRandomStep();
				}

			}
			pre_diff_map = diff_map;
		}
	} else
		ret = (++m_Consec_ErrClassifierRuns > LEARN_MAX_CONSECUTE_EQ_DIFFS);
    return ret;
}

void
Classifier::DoRandomStep() {
	double sign(0.), apply(0.);
	for(std::vector<double>::iterator w(m_W_q.begin()); w != m_W_q.end(); w++) {
		sign = ((rand() % 2) == 1 ? 1. : -.1);
		apply = ((rand() % 2) == 0 ? 1. : 0.);
		(*w) += sign * PSEUDO_COUNT * apply;
		LOG_TRAIN_LABELLING_2("rand> %.8f",(*w))
	}
}

void
Classifier::Make_ChunkWeights() {
	if(m_W_q_chunk.size() != 0) {
		std::fill(m_W_q.begin(),m_W_q.end(),0.);
		for(size_t i(0); i < m_W_q.size(); i++) {
			for(size_t j(0); j < m_W_q_chunk.size(); j++) {
				m_W_q[i] += m_W_q_chunk[j][i];
			}
			m_W_q[i] /= (double)m_W_q_chunk.size();
		}
		if(!m_Opt_No_EdgeModel_Learning) {
			std::fill(m_W_g.begin(),m_W_g.end(),0.);
			for(size_t i(0); i < m_W_g.size(); i++) {
				for(size_t j(0); j < m_W_g_chunk.size(); j++) {
					m_W_g[i] += m_W_g_chunk[j][i];
				}
				m_W_g[i] /= (double)m_W_q_chunk.size();
			}
			m_Curr_W_g = m_W_g;
			m_W_g_chunk.clear();
		}
		m_Curr_W_q = m_W_q;
		m_W_q_chunk.clear();
	}
}

bool
Classifier::ExecLearning(const std::string& indir, std::vector<GraphGen::Node<Types::Node> >& nodes,
							const std::vector<Types::Edge>& edges, const std::vector<GraphGen::Node<Types::Node>*>& c_alpha_backbone, size_t min_node_id, size_t max_node_id) {
	bool ret(false);

	m_Calpha_backbone = &c_alpha_backbone;
	m_Min_node_id = min_node_id;
	m_Max_node_id = max_node_id;

	if(!m_Hists_Initialized)
		Init(indir);
	if(MakeInitialMRF(nodes,edges)) {
		ret = Learn(nodes,edges);
	}

	CleanUp();
	return ret;
}

bool
Classifier::Learn(const std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges) {
	m_Prev_Diff_Map = 0.;
	m_LearnStep = 0;
	m_Prev_Structural_Diff_BigL = 0;
	m_Overall_LearnStep++;
	Eval::Init_ForObject(nodes.size());


	bool stop_learning(false);
	do {
		m_LearnStep++;
		stop_learning = LearnStep(nodes,edges);
	} while(!stop_learning && m_LearnStep < m_Opt_MaxNrLearnSteps_PerSample);
	if(m_Prv_W_q.size() != 0 && m_Prv_W_g.size() != 0)
		Eval::Save_Runs_BestWeightVectors(m_Prv_W_q,m_Prv_W_g);

	if(!stop_learning) {
		m_W_q_chunk.push_back(m_Prv_W_q);
		m_W_g_chunk.push_back(m_Prv_W_g);
	} else {
		m_W_q_chunk.push_back(m_W_q);
		m_W_g_chunk.push_back(m_W_g);
	}

	if(m_W_q_chunk.size() == m_Opt_Chunk_Size) {
		Make_ChunkWeights();
		Eval::Save_WeightVectors(false,true,m_W_q,m_W_g);
	} else {
		m_W_q = m_Curr_W_q;
		m_W_g = m_Curr_W_g;
	}
	return true;
}

void
Classifier::Post_Learning() {
	Make_ChunkWeights();
	Eval::Save_WeightVectors(true,false,m_W_q,m_W_g);
}

void
Classifier::MakeNodesQ(double& q_l0, double& q_l1, const GraphGen::Node<Types::Node>& n) {
	q_l0 = q_l1 = 0.;
	if(ConfigOpt_RASAext()) {
		q_l0 = m_W_q[index(n.m_Nd.m_RASA_noif_Slot,Types::Node::StateBinFeature_Feature_RASAnoif)] * n.m_Nd.m_RASA_noif_win;
		q_l1 = m_W_q[index(n.m_Nd.m_RASA_if_Slot,Types::Node::StateBinFeature_Feature_RASA_if)] * n.m_Nd.m_RASA_if_win;
	}
	if(ConfigOpt_Rg()) {
		q_l0 += m_W_q[index(n.m_Nd.m_Rg_noif_Slot,Types::Node::StateBinFeature_Feature_Rg_noif)] * n.m_Nd.m_Rg_noif;
		q_l1 += m_W_q[index(n.m_Nd.m_Rg_if_Slot,Types::Node::StateBinFeature_Feature_Rg_if)] * n.m_Nd.m_Rg_if;
	}
	if(ConfigOpt_Fe()) {
		q_l0 += m_W_q[index(n.m_Nd.m_Fe_noif_Slot,Types::Node::StateBinFeature_Feature_Fe_noif)] * n.m_Nd.m_Fe_noif;
		q_l1 += m_W_q[index(n.m_Nd.m_Fe_if_Slot,Types::Node::StateBinFeature_Feature_Fe_if)] * n.m_Nd.m_Fe_if;
	}
	if(ConfigOpt_ConservExt())  {
		q_l0 += m_W_q[index(n.m_Nd.m_Conserv_noif_Slot,Types::Node::StateBinFeature_Feature_CON_noif)] * n.m_Nd.m_Conserv_noif_win;
		q_l1 += m_W_q[index(n.m_Nd.m_Conserv_if_Slot,Types::Node::StateBinFeature_Feature_CON_if)] * n.m_Nd.m_Conserv_if_win;
	}
	if(ConfigOpt_PssmExt()) {
		q_l0 += m_W_q[index(n.m_Nd.m_PSSM_noif_Slot,Types::Node::StateBinFeature_Feature_PSSMnoif)] * n.m_Nd.m_PSSM_noif_win;
		q_l1 += m_W_q[index(n.m_Nd.m_PSSM_if_Slot,Types::Node::StateBinFeature_Feature_PSSM_if)] * n.m_Nd.m_PSSM_if_win;
	}
	if(ConfigOpt_Epros()) {
		q_l0 += m_W_q[index(n.m_Nd.m_EPROS_noif_Slot,Types::Node::StateBinFeature_Feature_Epros_noif)] * n.m_Nd.m_EPROS_noif_win;
		q_l1 += m_W_q[index(n.m_Nd.m_EPROS_if_Slot,Types::Node::StateBinFeature_Feature_Epros_if)] * n.m_Nd.m_EPROS_if_win;
	}
	if(ConfigOpt_Using_Exp_EnergyTerm()) {
		q_l0 = exp(q_l0);
		q_l1 = exp(q_l1);
	}
	if(q_l0 == 0. && q_l1 == 0.)
		q_l0 = q_l1 = PSEUDO_COUNT;
}

void
Classifier::MakeEdgeG(double& g_ii, double& g_ij) {
	g_ii = m_W_g[1] * m_Initial_g[1];
	g_ij = m_W_g[0] * m_Initial_g[0];
}

bool
Classifier::LearnStepComputations(const std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges, double S_curr_x_w, double S_x_w) {
	double diff(0), tau(0.);
	double b(0.), aa(0.);
	size_t rasa_ext_if(0), pssmext_if(0);
	size_t rasa_ext_noif(0), pssmext_noif(0);
	size_t conservext_if(0), conservext_noif(0);
	size_t epros_if(0), epros_noif(0);
	size_t rg_if(0), rg_noif(0);
	size_t fe_if(0), fe_noif(0);

	assert(m_FP_curr + m_FN_curr == m_Structural_Diff_BigL);
	b = (m_FP_Scale * (double)(m_FP_curr) + m_FN_Scale * (double)(m_FN_curr)) - (S_x_w - S_curr_x_w);
	assert(b >= 0.);

	fill(m_Diff.begin(),m_Diff.end(),0.);

	for(std::vector<GraphGen::Node<Types::Node> >::const_iterator n = nodes.begin(); n != nodes.end(); n++) {

		if(n->m_Id == -1)
			continue;

		rasa_ext_if = index(n->m_Nd.m_RASA_if_Slot,Types::Node::StateBinFeature_Feature_RASA_if);
		conservext_if = index(n->m_Nd.m_Conserv_if_Slot,Types::Node::StateBinFeature_Feature_CON_if);
		epros_if = index(n->m_Nd.m_EPROS_if_Slot,Types::Node::StateBinFeature_Feature_Epros_if);
		rg_if = index(n->m_Nd.m_Rg_if_Slot,Types::Node::StateBinFeature_Feature_Rg_if);
		fe_if = index(n->m_Nd.m_Fe_if_Slot,Types::Node::StateBinFeature_Feature_Fe_if);
		pssmext_if = index(n->m_Nd.m_PSSM_if_Slot,Types::Node::StateBinFeature_Feature_PSSM_if);

		rasa_ext_noif = index(n->m_Nd.m_RASA_noif_Slot,Types::Node::StateBinFeature_Feature_RASAnoif);
		conservext_noif = index(n->m_Nd.m_Conserv_noif_Slot,Types::Node::StateBinFeature_Feature_CON_noif);
		epros_noif = index(n->m_Nd.m_EPROS_noif_Slot,Types::Node::StateBinFeature_Feature_Epros_noif);
		rg_noif = index(n->m_Nd.m_Rg_noif_Slot,Types::Node::StateBinFeature_Feature_Rg_noif);
		fe_noif = index(n->m_Nd.m_Fe_noif_Slot,Types::Node::StateBinFeature_Feature_Fe_noif);
		pssmext_noif = index(n->m_Nd.m_PSSM_noif_Slot,Types::Node::StateBinFeature_Feature_PSSMnoif);


		if(ConfigOpt_RASAext()) {
			diff = m_Diff[rasa_ext_if] = (m_Initial_V[rasa_ext_if] - m_Curr_V[rasa_ext_if]);
			aa += (diff * diff);
			diff = m_Diff[rasa_ext_noif] = (m_Initial_V[rasa_ext_noif] - m_Curr_V[rasa_ext_noif]);
			aa += (diff * diff);
		}
		if(ConfigOpt_ConservExt()) {
			diff = m_Diff[conservext_if] = (m_Initial_V[conservext_if] - m_Curr_V[conservext_if]);
			aa += (diff * diff);
			diff = m_Diff[conservext_noif] = (m_Initial_V[conservext_noif] - m_Curr_V[conservext_noif]);
			aa += (diff * diff);
		}
		if(ConfigOpt_PssmExt()) {
			diff = m_Diff[pssmext_if] = (m_Initial_V[pssmext_if] - m_Curr_V[pssmext_if]);
			aa += (diff * diff);
			diff = m_Diff[pssmext_noif] = (m_Initial_V[pssmext_noif] - m_Curr_V[pssmext_noif]);
			aa += (diff * diff);
		}
		if(ConfigOpt_Epros()) {
			diff = m_Diff[epros_if] = (m_Initial_V[epros_if] - m_Curr_V[epros_if]);
			aa += (diff * diff);
			diff = m_Diff[epros_noif] = (m_Initial_V[epros_noif] - m_Curr_V[epros_noif]);
			aa += (diff * diff);
		}
		if(ConfigOpt_Rg()) {
			diff = m_Diff[rg_if] = (m_Initial_V[rg_if] - m_Curr_V[rg_if]);
			aa += (diff * diff);
			diff = m_Diff[rg_noif] = (m_Initial_V[rg_noif] - m_Curr_V[rg_noif]);
			aa += (diff * diff);
		}
		if(ConfigOpt_Fe()) {
			diff = m_Diff[fe_if] = (m_Initial_V[fe_if] - m_Curr_V[fe_if]);
			aa += (diff * diff);
			diff = m_Diff[fe_noif] = (m_Initial_V[fe_noif] - m_Curr_V[fe_noif]);
			aa += (diff * diff);
		}
	}

	if(!m_Opt_No_EdgeModel_Learning) {
		diff = (m_Initial_g_ij - m_Curr_g_ij);
		aa += (diff * diff);
		diff = (m_Initial_g_ii - m_Curr_g_ii);
		aa += (diff * diff);
	}
	if(aa == 0.)
		aa = 1.;
	tau = b/aa;
	for(size_t i(0); i < m_Diff.size(); i++) {
		LOG_TRAIN_LABELLING_4("-w%lu>%.8f %.4f",i,m_W_q[i],m_Diff[i])
		m_W_q[i] += m_Diff[i] * tau * LEARN_RATE_Q;
		if(ConfigOpt_SetWeightZero_iff_Negative() && m_W_q[i] < 0.)
			m_W_q[i] = 0.;
		LOG_TRAIN_LABELLING_3("+w%lu>%.8f",i,m_W_q[i])
	}

	if(!m_Opt_No_EdgeModel_Learning) {
		LOG_TRAIN_LABELLING_2("-wg0>%.8f",m_W_g[0])
		diff = LEARN_RATE_G * (m_Initial_g_ij - m_Curr_g_ij);
		m_W_g[0] += diff * tau;
		if(ConfigOpt_SetWeightZero_iff_Negative() && m_W_g[0] < 0.)
			m_W_g[0] = 0.;
		LOG_TRAIN_LABELLING_2("+wg0>%.8f",m_W_g[0])
		LOG_TRAIN_LABELLING_2("-wg1>%.8f",m_W_g[1])
		diff = LEARN_RATE_G * (m_Initial_g_ii - m_Curr_g_ii);
		m_W_g[1] += diff * tau;
		if(ConfigOpt_SetWeightZero_iff_Negative() && m_W_g[1] < 0.)
			m_W_g[1] = 0.;
		LOG_TRAIN_LABELLING_2("+wg1>%.8f",m_W_g[1])
	}
	UpdateFactors(nodes);
	return true;
}

bool
Classifier::Compute_Differences(const std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges) {
    m_Curr_g_ii = m_Curr_g_ij = 0;
    m_FP_curr = m_FN_curr = m_Structural_Diff_BigL = 0;
	fill(m_Curr_V.begin(),m_Curr_V.end(),0.);
	for(size_t j(0); j < m_MAP_Nodes.size(); j++) {
		std::vector<GraphGen::Node<Types::Node> >::const_iterator n = find_if(nodes.begin(),nodes.end(),Types::FindNodeId(m_Fg->var(j).label()));
		if(n->m_Id != -1) {
			if(m_Ref_Labelling[j] != m_MAP_Nodes[j]) {
				m_Structural_Diff_BigL++;
				if(m_Ref_Labelling[j] == 0)
					m_FP_curr++;
				else
					m_FN_curr++;
			}
			if(Eval::state(m_MAP_Nodes[j]) == Types::Node::RefLabel_If) {
				if(ConfigOpt_RASAext()) {
					m_Curr_V[index(n->m_Nd.m_RASA_if_Slot,Types::Node::StateBinFeature_Feature_RASA_if)] += n->m_Nd.m_RASA_if_win;
				}
				if(ConfigOpt_PssmExt()) {
					m_Curr_V[index(n->m_Nd.m_PSSM_if_Slot,Types::Node::StateBinFeature_Feature_PSSM_if)] += n->m_Nd.m_PSSM_if_win;
				}
				if(ConfigOpt_ConservExt()) {
					m_Curr_V[index(n->m_Nd.m_Conserv_if_Slot,Types::Node::StateBinFeature_Feature_CON_if)] += n->m_Nd.m_Conserv_if_win;
				}
				if(ConfigOpt_Epros()) {
					m_Curr_V[index(n->m_Nd.m_EPROS_if_Slot,Types::Node::StateBinFeature_Feature_Epros_if)] += n->m_Nd.m_EPROS_if_win;
				}
				if(ConfigOpt_Rg()) {
					m_Curr_V[index(n->m_Nd.m_Rg_if_Slot,Types::Node::StateBinFeature_Feature_Rg_if)] += n->m_Nd.m_Rg_if;
				}
				if(ConfigOpt_Fe()) {
					m_Curr_V[index(n->m_Nd.m_Fe_if_Slot,Types::Node::StateBinFeature_Feature_Fe_if)] += n->m_Nd.m_Fe_if;
				}
			} else {
				if(ConfigOpt_RASAext()) {
					m_Curr_V[index(n->m_Nd.m_RASA_noif_Slot,Types::Node::StateBinFeature_Feature_RASAnoif)] += n->m_Nd.m_RASA_noif_win;
				}
				if(ConfigOpt_PssmExt()) {
					m_Curr_V[index(n->m_Nd.m_PSSM_noif_Slot,Types::Node::StateBinFeature_Feature_PSSMnoif)] += n->m_Nd.m_PSSM_noif_win;
				}
				if(ConfigOpt_ConservExt()) {
					m_Curr_V[index(n->m_Nd.m_Conserv_noif_Slot,Types::Node::StateBinFeature_Feature_CON_noif)] += n->m_Nd.m_Conserv_noif_win;
				}
				if(ConfigOpt_Epros()) {
					m_Curr_V[index(n->m_Nd.m_EPROS_noif_Slot,Types::Node::StateBinFeature_Feature_Epros_noif)] += n->m_Nd.m_EPROS_noif_win;
				}
				if(ConfigOpt_Rg()) {
					m_Curr_V[index(n->m_Nd.m_Rg_noif_Slot,Types::Node::StateBinFeature_Feature_Rg_noif)] += n->m_Nd.m_Rg_noif;
				}
				if(ConfigOpt_Fe()) {
					m_Curr_V[index(n->m_Nd.m_Fe_noif_Slot,Types::Node::StateBinFeature_Feature_Fe_noif)] += n->m_Nd.m_Fe_noif;
				}
			}
		}
	}
    LOG_TRAIN_LABELLING_4("L>%lu %lu %lu",m_Structural_Diff_BigL,m_FP_curr,m_FN_curr)

    if(!m_Opt_No_EdgeModel_Learning) {
		for(size_t k(m_Nr_Nodes); k < m_Nr_Nodes + m_Nr_Edges; k++) {

			assert(m_Fg->factor(k).vars().size() == 2);
			assert(m_Fg->factor(k).nrStates() == 4);
			dai::SmallSet<size_t> node_indicies = m_Fg->findVars(m_Fg->factor(k).vars());
			assert(node_indicies.size() == 2);
			assert(node_indicies.front() < m_Nr_Nodes && node_indicies.back() < m_Nr_Nodes);
			std::vector<GraphGen::Node<Types::Node> >::const_iterator n1 = find_if(nodes.begin(),nodes.end(),Types::FindNodeId(m_Fg->var(node_indicies.front()).label()));
			std::vector<GraphGen::Node<Types::Node> >::const_iterator n2 = find_if(nodes.begin(),nodes.end(),Types::FindNodeId(m_Fg->var(node_indicies.back()).label()));
			if(m_MAP_Nodes[node_indicies.front()] == m_MAP_Nodes[node_indicies.back()])
				m_Curr_g_ii += m_Fg->factor(k).get(0);
			else
				m_Curr_g_ij += m_Fg->factor(k).get(1);
		}
    }
	return (m_Structural_Diff_BigL != 0);
}

bool
Classifier::Compute_Original_Li_Features(std::vector<GraphGen::Node<Types::Node> >& nodes) {
	bool ret(true);
	dai::Real fe_val(0.), rg_val(0.), rasa_val(0.), conserv_val(0.), pssm_val_if(0.), pssm_val_noif(0.), epros_val(0.);
	size_t slot(0);


	fill(m_Initial_V.begin(),m_Initial_V.end(),0.);
	for(std::vector<GraphGen::Node<Types::Node> >::iterator nd(nodes.begin()); nd != nodes.end(); nd++) {
		if(nd->m_Id != -1) {

			nd->m_Nd.m_RASA_if_win = 0.;
			nd->m_Nd.m_RASA_noif_win = 0.;
			nd->m_Nd.m_Conserv_if_win = 0.;
			nd->m_Nd.m_Conserv_noif_win = 0.;
			nd->m_Nd.m_PSSM_if_win = 0.;
			nd->m_Nd.m_PSSM_noif_win = 0.;
			nd->m_Nd.m_EPROS_if_win = 0.;
			nd->m_Nd.m_EPROS_noif_win = 0.;
			nd->m_Nd.m_Rg_if = 0.;
			nd->m_Nd.m_Rg_noif = 0.;
			nd->m_Nd.m_Fe_if = 0.;
			nd->m_Nd.m_Fe_noif = 0.;


			rasa_val = nd->m_Nd.m_Score_rASA;
			rg_val = nd->m_Nd.m_Score_Rg;
			fe_val = nd->m_Nd.m_Score_Fe;
			conserv_val = nd->m_Nd.m_Score_Conserv;
			epros_val = nd->m_Nd.m_Score_Epros;
			pssm_val_if = nd->m_Nd.m_Score_PSSM[Types::Node::ScorePssm_If];
			pssm_val_noif = nd->m_Nd.m_Score_PSSM[Types::Node::ScorePssm_NoIf];

			nd->m_Nd.m_RASA_if_win = if_rasa.GetProb(rasa_val,slot);
			nd->m_Nd.m_RASA_if_Slot = slot;
			nd->m_Nd.m_RASA_noif_win = noif_rasa.GetProb(rasa_val,slot);
			nd->m_Nd.m_RASA_noif_Slot = slot;
			nd->m_Nd.m_Conserv_if_win = if_conserv.GetProb(conserv_val,slot);
			nd->m_Nd.m_Conserv_if_Slot = slot;
			nd->m_Nd.m_Conserv_noif_win = noif_conserv.GetProb(conserv_val,slot);
			nd->m_Nd.m_Conserv_noif_Slot = slot;
			nd->m_Nd.m_PSSM_if_win = if_pssm.GetProb(pssm_val_if,slot);
			nd->m_Nd.m_PSSM_if_Slot = slot;
			nd->m_Nd.m_PSSM_noif_win = noif_pssm.GetProb(pssm_val_noif,slot);
			nd->m_Nd.m_PSSM_noif_Slot = slot;
			nd->m_Nd.m_EPROS_if_win = if_epros.GetProb(epros_val,slot);
			nd->m_Nd.m_EPROS_if_Slot = slot;
			nd->m_Nd.m_EPROS_noif_win = noif_epros.GetProb(epros_val,slot);
			nd->m_Nd.m_EPROS_noif_Slot = slot;
			nd->m_Nd.m_Rg_if = if_rg.GetProb(rg_val,slot);
			nd->m_Nd.m_Rg_if_Slot = slot;
			nd->m_Nd.m_Rg_noif = noif_rg.GetProb(rg_val,slot);
			nd->m_Nd.m_Rg_noif_Slot = slot;
			nd->m_Nd.m_Fe_if = if_fe.GetProb(fe_val,slot);
			nd->m_Nd.m_Fe_if_Slot = slot;
			nd->m_Nd.m_Fe_noif = noif_fe.GetProb(fe_val,slot);
			nd->m_Nd.m_Fe_noif_Slot = slot;


			if(nd->m_Nd.m_RefLabel == Types::Node::RefLabel_If) {
				if(ConfigOpt_RASAext()) {
					m_Initial_V[index(nd->m_Nd.m_RASA_if_Slot,Types::Node::StateBinFeature_Feature_RASA_if)] += nd->m_Nd.m_RASA_if_win;
				}
				if(ConfigOpt_PssmExt()) {
					m_Initial_V[index(nd->m_Nd.m_PSSM_if_Slot,Types::Node::StateBinFeature_Feature_PSSM_if)] += nd->m_Nd.m_PSSM_if_win;
				}
				if(ConfigOpt_ConservExt()) {
					m_Initial_V[index(nd->m_Nd.m_Conserv_if_Slot,Types::Node::StateBinFeature_Feature_CON_if)] += nd->m_Nd.m_Conserv_if_win;
				}
				if(ConfigOpt_Epros()) {
					m_Initial_V[index(nd->m_Nd.m_EPROS_if_Slot,Types::Node::StateBinFeature_Feature_Epros_if)] += nd->m_Nd.m_EPROS_if_win;
				}
				if(ConfigOpt_Rg()) {
					m_Initial_V[index(nd->m_Nd.m_Rg_if_Slot,Types::Node::StateBinFeature_Feature_Rg_if)] += nd->m_Nd.m_Rg_if;
				}
				if(ConfigOpt_Fe()) {
					m_Initial_V[index(nd->m_Nd.m_Fe_if_Slot,Types::Node::StateBinFeature_Feature_Fe_if)] += nd->m_Nd.m_Fe_if;
				}
			} else {
				if(ConfigOpt_RASAext()) {
					m_Initial_V[index(nd->m_Nd.m_RASA_noif_Slot,Types::Node::StateBinFeature_Feature_RASAnoif)] += nd->m_Nd.m_RASA_noif_win;
				}
				if(ConfigOpt_PssmExt()) {
					m_Initial_V[index(nd->m_Nd.m_PSSM_noif_Slot,Types::Node::StateBinFeature_Feature_PSSMnoif)] += nd->m_Nd.m_PSSM_noif_win;
				}
				if(ConfigOpt_ConservExt()) {
					m_Initial_V[index(nd->m_Nd.m_Conserv_noif_Slot,Types::Node::StateBinFeature_Feature_CON_noif)] += nd->m_Nd.m_Conserv_noif_win;
				}
				if(ConfigOpt_Epros()) {
					m_Initial_V[index(nd->m_Nd.m_EPROS_noif_Slot,Types::Node::StateBinFeature_Feature_Epros_noif)] += nd->m_Nd.m_EPROS_noif_win;
				}
				if(ConfigOpt_Rg()) {
					m_Initial_V[index(nd->m_Nd.m_Rg_noif_Slot,Types::Node::StateBinFeature_Feature_Rg_noif)] += nd->m_Nd.m_Rg_noif;
				}
				if(ConfigOpt_Fe()) {
					m_Initial_V[index(nd->m_Nd.m_Fe_noif_Slot,Types::Node::StateBinFeature_Feature_Fe_noif)] += nd->m_Nd.m_Fe_noif;
				}
			}
		}
	}
	m_Curr_W_q = m_W_q;
	return ret;
}

bool
Classifier::MakeInitialMRF(std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges) {

	bool ret(true);
	double g_ii(0.), g_ij(0.), q_l0(0.), q_l1(0.);
	m_Initial_g_ii = m_Initial_g_ij = m_Nr_Nodes = m_Nr_Edges = 0;
	m_TP_ref = 0;

	/* construct node/label assignments	*/
	Compute_Original_Li_Features(nodes);

	for(std::vector<GraphGen::Node<Types::Node> >::iterator nd(nodes.begin()); nd != nodes.end(); nd++) {
		if(nd->m_Id > -1) {
			dai::Var v_n((size_t)(*nd).m_Id,2);
			dai::Factor fac(v_n);

			MakeNodesQ(q_l0,q_l1,*nd);
			fac.set(0,q_l0);
			fac.set(1,q_l1);
			m_Factors.push_back(fac);

			if(MoreVerbose()) {
				bool cmp(Eval::state(nd->m_Nd.m_RefLabel) ? q_l1 > q_l0 : q_l0 > q_l1);
				LOG_TRAIN_LABELLING_7("+>%s%lu\t%.3f\t%.3f\t%1lu%1d",nd->m_Nd.m_Aa_Name.c_str(),nd->m_Id,q_l0,q_l1,Eval::state(nd->m_Nd.m_RefLabel),cmp)
			}
			++m_Nr_Nodes;
			m_Ref_Labelling.push_back(Eval::state((*nd).m_Nd.m_RefLabel));
			m_TP_ref += m_Ref_Labelling.back();
		}
	}
	LOG_TRAIN_LABELLING_2("\033[32m#if-%lu>\033[0m",m_TP_ref)

	/* construct edge assignments		*/
	for(std::vector<Types::Edge>::const_iterator e(edges.begin()); ret && e != edges.end(); e++) {
		/* search the partial factor graph for single node factors
		 *							*/
		if((*e).m_Node_1 != NULL && (*e).m_Node_2 != NULL) {

			std::vector<dai::Factor>::iterator n1 = find_if(m_Factors.begin(),m_Factors.end(),FindNode((*e).m_Node_1->m_Id));
			std::vector<dai::Factor>::iterator n2 = find_if(m_Factors.begin(),m_Factors.end(),FindNode((*e).m_Node_2->m_Id));

			/* Compute nodes' g-values per pair of labels
			 * 						*/
			if(n1 != m_Factors.end() && n2 != m_Factors.end()) {

				MakeEdgeG(g_ii,g_ij);
				if(MoreVerbose()) {
					LOG_TRAIN_LABELLING_9("+>%s%lu;%s%lu\t%.3f\t%.3f\t%.3f\t%.3f",(*e).m_Node_1->m_Nd.m_Aa_Name.c_str(),(*n1).vars().front().label(),
																				(*e).m_Node_2->m_Nd.m_Aa_Name.c_str(),(*n2).vars().front().label(),
																				g_ii,g_ij,g_ij,g_ii)
				}

				dai::VarSet vertices((*n1).vars().front(),(*n2).vars().front());
				dai::Factor edge(vertices);
				edge.set(0,g_ii);   // k_1 = 0, k_2 = 0
				edge.set(1,g_ij);   // k_1 = 1, k_2 = 0
				edge.set(2,g_ij);   // k_1 = 0, k_2 = 1
				edge.set(3,g_ii);   // k_1 = 1, k_2 = 1
				m_Factors.push_back(edge);
				++m_Nr_Edges;

				/* updates numbers of label pairs of current reference labelling
				 * 									*/
				if((*e).m_Node_1->m_Nd.m_RefLabel == (*e).m_Node_2->m_Nd.m_RefLabel) {
					m_Initial_g_ii += g_ii;
				} else {
					m_Initial_g_ij += g_ij;
				}
			} else {
				LOG_TRAIN_LABELLING_5("\033[41m WARN: edge wo. factornodes %s%lu;%s%lu\033[0m",(*e).m_Node_1->m_Nd.m_Aa_Name.c_str(),(*e).m_Node_1->m_Id,(*e).m_Node_2->m_Nd.m_Aa_Name.c_str(),(*e).m_Node_2->m_Id)
			}
		}
	}
	m_Fg_Needs_Update = true;
	return ret;
}

bool
Classifier::ExecValidation(const std::string& indir,std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges,
							const std::vector<GraphGen::Node<Types::Node>*>& c_alpha_backbone, size_t min_node_id, size_t max_node_id) {
	bool ret(false);
	double dummy(0.);

	m_Calpha_backbone = &c_alpha_backbone;
	m_Min_node_id = min_node_id;
	m_Max_node_id = max_node_id;

	if(!m_Hists_Initialized)
		Init(indir,false);
	if(MakeInitialMRF(nodes,edges)) {
		ret = RunClassifiers(nodes,edges,dummy,ConfigOpt_InferWithGibbs());
		Eval::EvalMAPLabelling(nodes);
	}
	CleanUp();
	return ret;
}

void
Classifier::StatsValidation() {
	Eval::StatsValidation();
}

bool
Classifier::GenerateModelGraphs(const std::string& indir,std::vector<GraphGen::Node<Types::Node> >& nodes, const std::vector<Types::Edge>& edges,
								const std::vector<GraphGen::Node<Types::Node>*>& c_alpha_backbone, size_t min_node_id, size_t max_node_id) {

	bool first_edge(true);
	m_Calpha_backbone = &c_alpha_backbone;
	m_Min_node_id = min_node_id;
	m_Max_node_id = max_node_id;

	if(!m_Hists_Initialized)
		Init(indir,false);
	if(MakeInitialMRF(nodes,edges)) {
	    std::ofstream os;
	    std::string filename;
	    filename = m_Output_Obj + ".mg";
	    os.open(filename.c_str());
	    if(os.is_open()) {
	        os.precision(5);
	        os << "numlabels:" << std::endl << "2" << std::endl << "nodes:" << std::endl;
		    for( size_t I = 0; I < m_Factors.size(); I++ ) {
		        if(m_Factors[I].vars().size() == 1) {
		        	assert(first_edge);
		        	os << m_Factors[I].vars().front().label() << "\t";
					for( size_t k = 0; k < m_Factors[I].nrStates(); k++ ) {
						os << m_Factors[I][k];
						if(k+1 != m_Factors[I].nrStates())
							os << "\t";
						else
							os << std::endl;
					}
		        }
		        if(m_Factors[I].vars().size() == 2) {
		        	if(first_edge) {
		        		first_edge = false;
		        		os << "edges:" << std::endl;
		        	}
		        	os << m_Factors[I].vars().front().label() << "\t";
		        	os << m_Factors[I].vars().back().label() << "\t";
					for( size_t k = 0; k < m_Factors[I].nrStates(); k++ ) {
						os << m_Factors[I][k];
						if(k+1 != m_Factors[I].nrStates())
							os << "\t";
						else
							os << std::endl;
					}
		        }
		    }
	        os.close();
	    } else {
	    	LOG_TRAIN_LABELLING_2("\033[41mCould not open file %s\033[0m",filename.c_str())
	    }
	}
	CleanUp();
	return true;
}

} // Training
